<?php
session_start();
require_once '../config/db.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];
$force_change = isset($_GET['force']) && $_GET['force'] == 1;

$success = "";
$error = "";

// Fetch current user info
$stmt = $conn->prepare("SELECT password FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($current_hashed_password);
$stmt->fetch();
$stmt->close();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $old_password = $_POST['old_password'];
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];

    // Validate old password
    if (!password_verify($old_password, $current_hashed_password)) {
        $error = "❌ Current password is incorrect.";
    } elseif ($new_password !== $confirm_password) {
        $error = "❌ New passwords do not match.";
    } elseif (password_verify($new_password, $current_hashed_password)) {
        $error = "⚠️ You cannot reuse the same password.";
    } else {
        $hashed_new = password_hash($new_password, PASSWORD_DEFAULT);
        $update = $conn->prepare("UPDATE users SET password = ?, password_changed_at = NOW() WHERE id = ?");
        $update->bind_param("si", $hashed_new, $user_id);
        if ($update->execute()) {
            $success = "✅ Password changed successfully.";
            if ($force_change) {
                header("Location: ../index.php");
                exit;
            }
        } else {
            $error = "❌ Failed to update password. Try again.";
        }
        $update->close();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Change Password | SmartPay360</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body {
      background: #f2f4f8;
      font-family: 'Segoe UI', sans-serif;
    }
    .change-box {
      max-width: 480px;
      margin: 70px auto;
      padding: 35px;
      background: #fff;
      border-radius: 12px;
      box-shadow: 0 0 15px rgba(0,0,0,0.1);
    }
    .form-control:focus {
      border-color: #006494;
      box-shadow: 0 0 0 0.2rem rgba(0,100,148,.25);
    }
    .btn-primary {
      background-color: #006494;
      border: none;
    }
    .btn-primary:hover {
      background-color: #004f73;
    }
  </style>
</head>
<body>
  <div class="change-box">
    <h4 class="text-center mb-4"><?= $force_change ? '🔐 Password Expired — Please Update' : 'Change Your Password' ?></h4>

    <?php if ($error): ?>
      <div class="alert alert-danger"><?= $error ?></div>
    <?php elseif ($success): ?>
      <div class="alert alert-success"><?= $success ?></div>
    <?php endif; ?>

    <form method="post" autocomplete="off">
      <div class="mb-3">
        <label class="form-label">Current Password</label>
        <input type="password" name="old_password" class="form-control" required autofocus>
      </div>
      <div class="mb-3">
        <label class="form-label">New Password</label>
        <input type="password" name="new_password" class="form-control" required>
      </div>
      <div class="mb-3">
        <label class="form-label">Confirm New Password</label>
        <input type="password" name="confirm_password" class="form-control" required>
      </div>
      <button type="submit" class="btn btn-primary w-100">Update Password</button>
    </form>

    <?php if (!$force_change): ?>
      <div class="text-center mt-3">
        <a href="../index.php" class="text-decoration-none">← Back to Dashboard</a>
      </div>
    <?php endif; ?>
  </div>
</body>
</html>
