<?php
// analytics_sales_purchases_payments.php
// Combined analytics: Sales vs Purchases vs Payments (and Petty Cash)
// ---------------------------------------------------------------

session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// ===== Helpers =====
function fmt($n) { return number_format((float)$n, 2); }
function bindAndExec(mysqli $conn, string $sql, string $types = '', array $params = []) {
    $stmt = $conn->prepare($sql);
    if (!$stmt) { die("SQL error: " . $conn->error); }
    if ($types !== '' && !empty($params)) { $stmt->bind_param($types, ...$params); }
    $stmt->execute();
    return $stmt->get_result();
}

// ===== Filters =====
$today = date('Y-m-d');
$from = $_GET['from'] ?? date('Y-m-01');
$to   = $_GET['to']   ?? date('Y-m-d');
$branch_id = isset($_GET['branch_id']) ? (int)$_GET['branch_id'] : 0;
$supplier_id = isset($_GET['supplier_id']) ? (int)$_GET['supplier_id'] : 0;

$branches_rs = $conn->query("SELECT id, branch_name FROM branches ORDER BY branch_name");
$suppliers_rs = $conn->query("SELECT id, name FROM suppliers ORDER BY name");

// Build optional WHERE fragments + params for each domain
$baseTypes = 'ss';
$baseParams = [$from, $to];

// --- SALES (Till Reports) ---
$salesTypes = $baseTypes;
$salesParams = $baseParams;
$salesBranchWhere = '';
if ($branch_id > 0) {
    $salesBranchWhere = " AND h.branch_id = ? ";
    $salesTypes .= 'i';
    $salesParams[] = $branch_id;
}

// Derived cheques per report to avoid fan-out
$salesSQL = "
    WITH cheque_sum AS (
        SELECT report_id, SUM(amount) AS cheques_total
        FROM till_report_cheques
        GROUP BY report_id
    )
    SELECT 
        DATE(h.report_date) AS dte,
        b.branch_name,
        SUM(
          (d.note_1000 + d.note_500 + d.note_200 + d.note_100 + d.note_50)
          + d.coins
          + (d.drop1 + d.drop2 + d.drop3)
        ) AS cash_total,
        SUM(d.mpesa + d.stk) AS mpesa_total,
        SUM(d.visa) AS visa_total,
        SUM(d.credit) AS credit_total,
        SUM(d.declared_total) AS declared_total,
        SUM(COALESCE(c.cheques_total,0)) AS cheques_total
    FROM till_report_headers h
    JOIN branches b ON b.id = h.branch_id
    JOIN till_report_details d ON d.report_id = h.id
    LEFT JOIN cheque_sum c ON c.report_id = h.id
    WHERE h.report_date BETWEEN ? AND ?
    $salesBranchWhere
    GROUP BY DATE(h.report_date), b.branch_name
    ORDER BY dte ASC, b.branch_name ASC
";
$salesRes = bindAndExec($conn, $salesSQL, $salesTypes, $salesParams);

// --- PURCHASES (Supplier Invoices) ---
$purTypes = $baseTypes;
$purParams = $baseParams;
$purSuppWhere = '';
if ($supplier_id > 0) {
    $purSuppWhere .= " AND i.supplier_id = ? ";
    $purTypes .= 'i';
    $purParams[] = $supplier_id;
}
// Optional: if invoices table has branch_id, uncomment both lines below to enable branch filter for purchases
// if ($branch_id > 0) { $purSuppWhere .= " AND i.branch_id = ? "; $purTypes .= 'i'; $purParams[] = $branch_id; }

$purchaseSQL = "
    SELECT DATE(i.invoice_date) AS dte, SUM(i.total_amount) AS purchases_total
    FROM invoices i
    WHERE i.invoice_date BETWEEN ? AND ?
    $purSuppWhere
    GROUP BY DATE(i.invoice_date)
    ORDER BY dte ASC
";
$purchaseRes = bindAndExec($conn, $purchaseSQL, $purTypes, $purParams);

// --- PAYMENTS (to Suppliers) ---
$payTypes = $baseTypes;
$payParams = $baseParams;
$payWhere = '';
if ($supplier_id > 0) { $payWhere .= " AND p.supplier_id = ? "; $payTypes .= 'i'; $payParams[] = $supplier_id; }

$paymentSQL = "
    SELECT DATE(p.payment_date) AS dte, SUM(p.amount) AS payments_total
    FROM payments p
    WHERE p.payment_date BETWEEN ? AND ?
    $payWhere
    GROUP BY DATE(p.payment_date)
    ORDER BY dte ASC
";
$paymentRes = bindAndExec($conn, $paymentSQL, $payTypes, $payParams);

// --- PETTY CASH ---
$pcTypes = $baseTypes;
$pcParams = $baseParams;
$pcWhere = '';
if ($branch_id > 0) { $pcWhere .= " AND pct.branch_id = ? "; $pcTypes .= 'i'; $pcParams[] = $branch_id; }

$pettySQL = "
    SELECT DATE(pct.transaction_date) AS dte, SUM(pct.amount) AS petty_total
    FROM petty_cash_transactions pct
    WHERE pct.transaction_date BETWEEN ? AND ?
    $pcWhere
    GROUP BY DATE(pct.transaction_date)
    ORDER BY dte ASC
";
$pettyRes = bindAndExec($conn, $pettySQL, $pcTypes, $pcParams);

// ===== Merge timeseries by date =====
$dateset = [];
$totals = [
    'sales_actual' => 0, 'sales_declared' => 0,
    'purchases' => 0, 'payments' => 0, 'petty' => 0
];

// Sales rows (build both actual & declared)
while ($r = $salesRes->fetch_assoc()) {
    $d = $r['dte'];
    if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
    $actual = (float)$r['cash_total'] + (float)$r['mpesa_total'] + (float)$r['visa_total'] + (float)$r['cheques_total'] + (float)$r['credit_total'];
    $declared = (float)$r['declared_total'];
    $dateset[$d]['sales_actual'] += $actual;
    $dateset[$d]['sales_declared'] += $declared;
    $totals['sales_actual'] += $actual;
    $totals['sales_declared'] += $declared;
}

// Purchases
while ($r = $purchaseRes->fetch_assoc()) {
    $d = $r['dte'];
    if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
    $val = (float)$r['purchases_total'];
    $dateset[$d]['purchases'] += $val;
    $totals['purchases'] += $val;
}

// Payments
while ($r = $paymentRes->fetch_assoc()) {
    $d = $r['dte'];
    if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
    $val = (float)$r['payments_total'];
    $dateset[$d]['payments'] += $val;
    $totals['payments'] += $val;
}

// Petty cash
while ($r = $pettyRes->fetch_assoc()) {
    $d = $r['dte'];
    if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
    $val = (float)$r['petty_total'];
    $dateset[$d]['petty'] += $val;
    $totals['petty'] += $val;
}

// Sort by date
ksort($dateset);

// KPIs
$gross_margin_proxy = $totals['sales_declared'] - $totals['purchases']; // proxy
$cash_gap = $totals['sales_actual'] - ($totals['payments'] + $totals['petty']); // how much inflow vs outflow

// Data for Chart.js
$labels = array_keys($dateset);
$salesActualData = array_map(fn($d)=> round($d['sales_actual'],2), $dateset);
$salesDeclaredData = array_map(fn($d)=> round($d['sales_declared'],2), $dateset);
$purchaseData = array_map(fn($d)=> round($d['purchases'],2), $dateset);
$paymentData = array_map(fn($d)=> round($d['payments'],2), $dateset);
$pettyData = array_map(fn($d)=> round($d['petty'],2), $dateset);
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Analytics — Sales vs Purchases vs Payments</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- BS + Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
  <!-- Chart.js -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
  <style>
    body { background:#f5f7fb; }
    .kpi-card { border:0; border-radius: 14px; box-shadow: 0 6px 18px rgba(0,0,0,.06); }
    .kpi-card .card-body { padding: 16px 18px; }
    .kpi-value { font-weight: 800; font-size: 1.25rem; }
    .kpi-label { font-size: .85rem; color:#6c757d; }
    .chip { font-size:.75rem; padding:.15rem .5rem; border-radius: 10px; }
    .table-sticky thead th { position: sticky; top: 0; background: #fff; z-index: 1; }
    .mini-help { font-size: 12px; color:#6c757d; }
    @media print { .no-print { display:none; } body{ background:#fff; } }
  </style>
</head>
<body>
<div class="container-fluid py-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h4 class="mb-0 fw-bold text-primary"><i class="fa-solid fa-chart-line me-2"></i>Sales vs Purchases vs Payments</h4>
    <div class="no-print">
      <button onclick="window.print()" class="btn btn-outline-dark btn-sm"><i class="fas fa-print me-1"></i> Print</button>
    </div>
  </div>

  <!-- Filters -->
  <div class="card shadow-sm mb-4">
    <div class="card-body">
      <form method="GET" class="row g-2 align-items-end">
        <div class="col-md-2">
          <label class="form-label">From</label>
          <input type="date" name="from" value="<?= htmlspecialchars($from) ?>" class="form-control form-control-sm">
        </div>
        <div class="col-md-2">
          <label class="form-label">To</label>
          <input type="date" name="to" value="<?= htmlspecialchars($to) ?>" class="form-control form-control-sm">
        </div>
        <div class="col-md-3">
          <label class="form-label">Branch</label>
          <select name="branch_id" class="form-select form-select-sm">
            <option value="">-- All Branches --</option>
            <?php while ($b = $branches_rs->fetch_assoc()): ?>
              <option value="<?= $b['id'] ?>" <?= $branch_id == $b['id'] ? 'selected' : '' ?>>
                <?= htmlspecialchars($b['branch_name']) ?>
              </option>
            <?php endwhile; ?>
          </select>
          <div class="mini-help">Affects Sales & Petty Cash. (Toggle for Purchases if invoices have branch_id.)</div>
        </div>
        <div class="col-md-3">
          <label class="form-label">Supplier</label>
          <select name="supplier_id" class="form-select form-select-sm">
            <option value="">-- All Suppliers --</option>
            <?php while ($s = $suppliers_rs->fetch_assoc()): ?>
              <option value="<?= $s['id'] ?>" <?= $supplier_id == $s['id'] ? 'selected' : '' ?>>
                <?= htmlspecialchars($s['name']) ?>
              </option>
            <?php endwhile; ?>
          </select>
          <div class="mini-help">Filters Purchases & Payments only.</div>
        </div>
        <div class="col-md-2">
          <button class="btn btn-primary btn-sm w-100"><i class="fas fa-filter me-1"></i> Apply</button>
        </div>
      </form>
    </div>
  </div>

  <!-- KPIs -->
  <div class="row g-3 mb-4">
    <div class="col-md-2">
      <div class="card kpi-card">
        <div class="card-body">
          <div class="kpi-label">Sales (Actual)</div>
          <div class="kpi-value text-success">KSh <?= fmt($totals['sales_actual']) ?></div>
          <span class="chip bg-light text-success"><i class="fa-solid fa-circle me-1"></i> Cash+MPESA+VISA+Chq+Credit</span>
        </div>
      </div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card">
        <div class="card-body">
          <div class="kpi-label">Sales (POS Expected)</div>
          <div class="kpi-value text-primary">KSh <?= fmt($totals['sales_declared']) ?></div>
          <span class="chip bg-light text-primary"><i class="fa-regular fa-rectangle-list me-1"></i> Declared</span>
        </div>
      </div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card">
        <div class="card-body">
          <div class="kpi-label">Purchases</div>
          <div class="kpi-value text-danger">KSh <?= fmt($totals['purchases']) ?></div>
          <span class="chip bg-light text-danger"><i class="fa-solid fa-file-invoice me-1"></i> Invoices</span>
        </div>
      </div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card">
        <div class="card-body">
          <div class="kpi-label">Payments</div>
          <div class="kpi-value text-warning">KSh <?= fmt($totals['payments']) ?></div>
          <span class="chip bg-light text-warning"><i class="fa-solid fa-money-bill-transfer me-1"></i> Paid to Suppliers</span>
        </div>
      </div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card">
        <div class="card-body">
          <div class="kpi-label">Petty Cash</div>
          <div class="kpi-value text-secondary">KSh <?= fmt($totals['petty']) ?></div>
          <span class="chip bg-light text-secondary"><i class="fa-solid fa-sack-dollar me-1"></i> Expenses</span>
        </div>
      </div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card">
        <div class="card-body">
          <div class="kpi-label">Gross Margin (Proxy)</div>
          <div class="kpi-value <?= $gross_margin_proxy >=0 ? 'text-success' : 'text-danger' ?>">KSh <?= fmt($gross_margin_proxy) ?></div>
          <span class="chip bg-light"><i class="fa-solid fa-scale-balanced me-1"></i> Sales(Declared) - Purchases</span>
        </div>
      </div>
    </div>
  </div>

  <!-- Big Combo Chart -->
  <div class="card shadow-sm mb-4">
    <div class="card-body">
      <div class="d-flex justify-content-between align-items-center mb-2">
        <h6 class="mb-0 fw-bold"><i class="fa-solid fa-chart-column me-2"></i>Daily Trend</h6>
        <span class="mini-help">Bars: Sales Actual & Declared | Lines: Purchases, Payments, Petty Cash</span>
      </div>
      <canvas id="comboChart" height="120"></canvas>
    </div>
  </div>

  <!-- Data Table -->
  <div class="card shadow-sm">
    <div class="card-body table-responsive">
      <table class="table table-sm table-bordered table-hover align-middle table-sticky mb-0">
        <thead class="table-light text-center">
          <tr>
            <th>Date</th>
            <th>Sales (Actual)</th>
            <th>Sales (Declared)</th>
            <th>Purchases</th>
            <th>Payments</th>
            <th>Petty Cash</th>
            <th>Cash Gap</th>
            <th>Variance (Actual - Declared)</th>
          </tr>
        </thead>
        <tbody class="text-center">
          <?php if (empty($dateset)): ?>
            <tr><td colspan="8" class="text-muted">No data for selected filters.</td></tr>
          <?php else: ?>
            <?php foreach ($dateset as $d => $vals): 
              $cash_gap_row = $vals['sales_actual'] - ($vals['payments'] + $vals['petty']);
              $variance_row = $vals['sales_actual'] - $vals['sales_declared'];
            ?>
              <tr>
                <td><?= htmlspecialchars($d) ?></td>
                <td class="text-success fw-semibold">KSh <?= fmt($vals['sales_actual']) ?></td>
                <td class="text-primary">KSh <?= fmt($vals['sales_declared']) ?></td>
                <td class="text-danger">KSh <?= fmt($vals['purchases']) ?></td>
                <td class="text-warning">KSh <?= fmt($vals['payments']) ?></td>
                <td class="text-secondary">KSh <?= fmt($vals['petty']) ?></td>
                <td class="<?= $cash_gap_row>=0?'text-success':'text-danger' ?> fw-semibold">KSh <?= fmt($cash_gap_row) ?></td>
                <td class="<?= $variance_row==0?'text-muted':($variance_row>0?'text-success':'text-danger') ?>">
                  <?= $variance_row>0?'+':'' ?>KSh <?= fmt($variance_row) ?>
                </td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
        </tbody>
        <?php if (!empty($dateset)): ?>
        <tfoot class="table-light">
          <tr class="text-center fw-bold">
            <td>Totals</td>
            <td class="text-success">KSh <?= fmt($totals['sales_actual']) ?></td>
            <td class="text-primary">KSh <?= fmt($totals['sales_declared']) ?></td>
            <td class="text-danger">KSh <?= fmt($totals['purchases']) ?></td>
            <td class="text-warning">KSh <?= fmt($totals['payments']) ?></td>
            <td class="text-secondary">KSh <?= fmt($totals['petty']) ?></td>
            <td class="<?= $cash_gap>=0?'text-success':'text-danger' ?>"><?= ($cash_gap>=0?'+':'') ?>KSh <?= fmt($cash_gap) ?></td>
            <td class="<?= ($totals['sales_actual']-$totals['sales_declared'])>=0?'text-success':'text-danger' ?>">
              <?= ($totals['sales_actual']-$totals['sales_declared'])>0?'+':'' ?>KSh <?= fmt($totals['sales_actual']-$totals['sales_declared']) ?>
            </td>
          </tr>
        </tfoot>
        <?php endif; ?>
      </table>
    </div>
  </div>

</div>

<!-- Scripts -->
<script>
const labels = <?= json_encode($labels) ?>;
const salesActual = <?= json_encode($salesActualData) ?>;
const salesDeclared = <?= json_encode($salesDeclaredData) ?>;
const purchases = <?= json_encode($purchaseData) ?>;
const payments = <?= json_encode($paymentData) ?>;
const petty = <?= json_encode($pettyData) ?>;

const ctx = document.getElementById('comboChart');
const comboChart = new Chart(ctx, {
  type: 'bar',
  data: {
    labels: labels,
    datasets: [
      { label: 'Sales (Actual)', data: salesActual, yAxisID: 'y', order: 2 },
      { label: 'Sales (Declared)', data: salesDeclared, yAxisID: 'y', order: 2 },
      { label: 'Purchases', data: purchases, type: 'line', yAxisID: 'y', order: 1, tension: .25 },
      { label: 'Payments', data: payments, type: 'line', yAxisID: 'y', order: 1, tension: .25 },
      { label: 'Petty Cash', data: petty, type: 'line', yAxisID: 'y', order: 1, tension: .25 }
    ]
  },
  options: {
    responsive: true,
    maintainAspectRatio: false,
    interaction: { mode: 'index', intersect: false },
    plugins: {
      legend: { position: 'top' },
      tooltip: { callbacks: { label: (ctx) => `${ctx.dataset.label}: KSh ${Number(ctx.parsed.y).toLocaleString()}` } }
    },
    scales: {
      y: {
        beginAtZero: true,
        ticks: { callback: (value)=> 'KSh ' + value.toLocaleString() },
        title: { display: true, text: 'Amount (KES)' }
      },
      x: { ticks: { maxRotation: 0, autoSkip: true } }
    }
  }
});
</script>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<?php include '../../includes/footer.php'; ?>
</body>
</html>
