<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

$loading_id = $_GET['loading_id'] ?? null;
if (!$loading_id) {
    echo "<div class='alert alert-danger'>⚠️ No loading selected.</div>";
    exit;
}

// Fetch loading session
$stmt = $conn->prepare("
    SELECT l.*, v.vehicle_no, r.name AS route_name, v.driver_name
    FROM loadings l
    JOIN vehicles v ON l.vehicle_id = v.id
    JOIN routes r ON l.route_id = r.id
    WHERE l.id = ?
");
$stmt->bind_param("i", $loading_id);
$stmt->execute();
$loading = $stmt->get_result()->fetch_assoc();
if (!$loading) {
    echo "<div class='alert alert-danger'>⚠️ Loading session not found.</div>";
    exit;
}

// Fetch loaded items
$stmt = $conn->prepare("
    SELECT li.sku_id, s.name AS sku_name, li.crates_loaded, s.loaves_per_crate, s.price_per_crate
    FROM loading_items li
    JOIN skus s ON li.sku_id = s.id
    WHERE li.loading_id = ?
");
$stmt->bind_param("i", $loading_id);
$stmt->execute();
$loaded_items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Fetch returns
$stmt = $conn->prepare("SELECT sku_id, returned_crates, returned_pieces FROM returns WHERE delivery_id = ?");
$stmt->bind_param("i", $loading_id);
$stmt->execute();
$returned_data = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

$returns = [];
foreach ($returned_data as $r) {
    $returns[$r['sku_id']] = [
        'crates' => $r['returned_crates'],
        'pieces' => $r['returned_pieces']
    ];
}

// Fetch payment
$stmt = $conn->prepare("
    SELECT cash_amount, mpesa_amount, cheque_amount, cheque_no, remarks, payment_date
    FROM bread_payments
    WHERE loading_id = ?
    ORDER BY id DESC LIMIT 1
");
$stmt->bind_param("i", $loading_id);
$stmt->execute();
$payment = $stmt->get_result()->fetch_assoc();
$stmt->close();
$total_paid = $payment ? ($payment['cash_amount'] + $payment['mpesa_amount'] + $payment['cheque_amount']) : 0;

// Totals
$grand_total = 0;
$total_sold_loaves = 0;
$total_returned_loaves = 0;
$total_deficit = 0;
$sku_labels = [];
$sku_values = [];

$mileage_start = (int)$loading['mileage_start'];
$mileage_end = (int)$loading['mileage_end'];
$distance_covered = ($mileage_end > $mileage_start) ? ($mileage_end - $mileage_start) : 0;
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>📊 Bread Loading Summary Report</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <style>
    @media print {
        .no-print { display: none; }
    }
  </style>
</head>
<body>
<div class="container my-4">

  <div class="d-flex justify-content-between align-items-center mb-4">
    <h4 class="text-primary">📊 Bread Loading Summary Report</h4>
    <button onclick="window.print()" class="btn btn-outline-dark btn-sm no-print">🖨️ Print</button>
  </div>

  <div class="card mb-4">
    <div class="card-body">
      <p><strong>Vehicle:</strong> <?= htmlspecialchars($loading['vehicle_no']) ?> |
         <strong>Driver:</strong> <?= htmlspecialchars($loading['driver_name']) ?> |
         <strong>Route:</strong> <?= htmlspecialchars($loading['route_name']) ?> |
         <strong>Date:</strong> <?= htmlspecialchars($loading['loading_date']) ?></p>
      <p><strong>Mileage Start:</strong> <?= $mileage_start ?> km &nbsp;&nbsp;
         <strong>Mileage End:</strong> <?= $mileage_end ?> km &nbsp;&nbsp;
         <strong>Distance Covered:</strong> <?= $distance_covered ?> km</p>

      <hr>
      <h6 class="text-dark fw-bold mb-2">💰 Payment Summary</h6>
      <?php if ($payment): ?>
      <ul class="list-group list-group-flush">
        <li class="list-group-item">🧾 <strong>Payment Date:</strong> <?= htmlspecialchars($payment['payment_date']) ?></li>
        <li class="list-group-item">💵 <strong>Cash:</strong> Ksh <?= number_format($payment['cash_amount'], 2) ?></li>
        <li class="list-group-item">📲 <strong>MPESA:</strong> Ksh <?= number_format($payment['mpesa_amount'], 2) ?></li>
        <li class="list-group-item">🧾 <strong>Cheque:</strong> Ksh <?= number_format($payment['cheque_amount'], 2) ?>
            <?php if (!empty($payment['cheque_no'])): ?>
                <small class="text-muted">(Cheque No: <?= htmlspecialchars($payment['cheque_no']) ?>)</small>
            <?php endif; ?>
        </li>
        <?php if (!empty($payment['remarks'])): ?>
            <li class="list-group-item">📝 <strong>Remarks:</strong> <?= htmlspecialchars($payment['remarks']) ?></li>
        <?php endif; ?>
        <li class="list-group-item fw-bold bg-light">✅ <strong>Total Paid:</strong> Ksh <?= number_format($total_paid, 2) ?></li>
      </ul>
      <?php else: ?>
      <div class="alert alert-warning">⚠️ No payment recorded yet for this delivery.</div>
      <?php endif; ?>
    </div>
  </div>

  <div class="table-responsive mb-4">
    <table class="table table-bordered table-striped align-middle">
      <thead class="table-light">
        <tr>
          <th>SKU</th>
          <th>Loaded</th>
          <th>Returned</th>
          <th>Sold (pcs)</th>
          <th>Deficit (pcs)</th>
          <th>Unit Price</th>
          <th>Value (Ksh)</th>
          <th>% Sold</th>
          <th>% Returned</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($loaded_items as $item):
            $sku_id = $item['sku_id'];
            $ret = $returns[$sku_id] ?? ['crates' => 0, 'pieces' => 0];

            $loaded_crates = (int)$item['crates_loaded'];
            $loaves_per_crate = (int)$item['loaves_per_crate'];
            $loaded_loaves = $loaded_crates * $loaves_per_crate;

            $returned_loaves = ((int)$ret['crates'] * $loaves_per_crate) + ((int)$ret['pieces']);
            $sold_loaves = $loaded_loaves - $returned_loaves;
            $deficit_loaves = $loaded_loaves - ($sold_loaves + $returned_loaves);

            $value = ($sold_loaves / $loaves_per_crate) * $item['price_per_crate'];
            $sold_pct = $loaded_loaves > 0 ? round(($sold_loaves / $loaded_loaves) * 100, 1) : 0;
            $returned_pct = 100 - $sold_pct;

            $grand_total += $value;
            $total_sold_loaves += $sold_loaves;
            $total_returned_loaves += $returned_loaves;
            $total_deficit += $deficit_loaves;

            $sku_labels[] = $item['sku_name'];
            $sku_values[] = round($value, 2);
        ?>
        <tr>
          <td><?= htmlspecialchars($item['sku_name']) ?></td>
          <td class="text-center"><?= $loaded_crates ?> crates<br><small><?= $loaded_loaves ?> pcs</small></td>
          <td class="text-center"><?= $ret['crates'] ?? 0 ?> crates<br><small><?= $ret['pieces'] ?? 0 ?> pcs</small></td>
          <td class="text-center"><?= $sold_loaves ?> pcs</td>
          <td class="text-center <?= $deficit_loaves > 0 ? 'text-danger' : 'text-success' ?>"><?= $deficit_loaves ?></td>
          <td class="text-end"><?= number_format($item['price_per_crate'], 2) ?></td>
          <td class="text-end fw-bold"><?= number_format($value, 2) ?></td>
          <td class="text-center text-success"><?= $sold_pct ?>%</td>
          <td class="text-center text-danger"><?= $returned_pct ?>%</td>
        </tr>
        <?php endforeach; ?>
        <tr class="table-dark fw-bold">
          <td colspan="3" class="text-end">TOTAL</td>
          <td class="text-center"><?= $total_sold_loaves ?> pcs</td>
          <td class="text-center text-danger"><?= $total_deficit ?> pcs</td>
          <td></td>
          <td class="text-end">Ksh <?= number_format($grand_total, 2) ?></td>
          <td colspan="2"></td>
        </tr>
      </tbody>
    </table>
  </div>

  <div class="card">
    <div class="card-body">
      <h6 class="text-muted">📈 Sales Distribution by SKU</h6>
      <canvas id="salesChart" height="100"></canvas>
    </div>
  </div>

</div>

<!-- Chart JS -->
<script>
const ctx = document.getElementById('salesChart');
new Chart(ctx, {
    type: 'doughnut',
    data: {
        labels: <?= json_encode($sku_labels) ?>,
        datasets: [{
            data: <?= json_encode($sku_values) ?>,
            backgroundColor: [
                '#4e73df', '#1cc88a', '#36b9cc', '#f6c23e',
                '#e74a3b', '#858796', '#2e59d9', '#17a673', '#f4b619'
            ],
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: { position: 'right' },
            title: { display: false }
        }
    }
});
</script>

</body>
</html>
