<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// Fetch available loadings
$loadings_query = $conn->query("
    SELECT l.id, v.plate_number AS vehicle_no, r.name AS route_name, l.loading_date
    FROM loadings l
    JOIN vehicles v ON l.vehicle_id = v.id
    JOIN routes r ON l.route_id = r.id
    ORDER BY l.loading_date DESC
");

if (!$loadings_query) {
    die("Loadings query failed: " . $conn->error);
}

$loadings = $loadings_query->fetch_all(MYSQLI_ASSOC);

$selected_id = $_GET['loading_id'] ?? null;
$loading = null;
$items = [];

if ($selected_id) {
    $stmt = $conn->prepare("
        SELECT l.*, v.plate_number AS vehicle_no, v.driver_name, r.name AS route_name
        FROM loadings l
        JOIN vehicles v ON l.vehicle_id = v.id
        JOIN routes r ON l.route_id = r.id
        WHERE l.id = ?
    ");
    if ($stmt) {
        $stmt->bind_param('i', $selected_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $loading = $result->fetch_assoc();

        if ($loading) {
            $stmt_items = $conn->prepare("
                SELECT li.*, s.name AS sku_name, s.price_per_crate, s.loaves_per_crate
                FROM loading_items li
                JOIN skus s ON li.sku_id = s.id
                WHERE li.loading_id = ?
            ");
            $stmt_items->bind_param('i', $selected_id);
            $stmt_items->execute();
            $items = $stmt_items->get_result()->fetch_all(MYSQLI_ASSOC);
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>🍞 Record Bread Returns</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .sticky-footer {
        position: sticky;
        bottom: 0;
        background: #f8f9fa;
        padding: 10px;
        border-top: 2px solid #dee2e6;
        z-index: 10;
    }
    .is-invalid {
        border-color: red !important;
    }
  </style>
</head>

<body>
<div class="main-content container py-4">

<h4 class="fw-bold text-primary mb-4"><i class="fas fa-undo me-2"></i> Record Bread Returns</h4>

<!-- Select loading session -->
<form method="GET" class="row g-3 mb-4">
    <div class="col-md-6">
        <label class="form-label">Select Loading:</label>
        <select name="loading_id" class="form-select" onchange="this.form.submit()" required>
            <option value="">-- Choose Loading --</option>
            <?php foreach ($loadings as $l): ?>
                <option value="<?= $l['id'] ?>" <?= ($selected_id == $l['id']) ? 'selected' : '' ?>>
                    <?= htmlspecialchars("ID#{$l['id']} - {$l['vehicle_no']} ({$l['route_name']}) {$l['loading_date']}") ?>
                </option>
            <?php endforeach; ?>
        </select>
    </div>
</form>

<?php if ($selected_id && !$loading): ?>
    <div class="alert alert-danger">⚠️ Loading not found!</div>
<?php endif; ?>

<?php if ($loading): ?>
    <form method="POST" action="save_returns.php" onsubmit="return confirm('Save returns and proceed?')">
        <input type="hidden" name="loading_id" value="<?= $loading['id'] ?>">

        <div class="mb-3">
            <h5>🚚 <?= htmlspecialchars($loading['vehicle_no']) ?> | Driver: <?= htmlspecialchars($loading['driver_name']) ?> | Route: <?= htmlspecialchars($loading['route_name']) ?></h5>
        </div>

        <div class="table-responsive">
            <table class="table table-bordered table-sm align-middle">
                <thead class="table-light">
                    <tr>
                        <th>SKU</th>
                        <th>Crates Loaded</th>
                        <th>Returned (Crates)</th>
                        <th>Returned (Pieces)</th>
                        <th>Sold (Pieces)</th>
                        <th>Value Sold (Ksh)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($items as $row): ?>
                    <tr>
                        <td><?= htmlspecialchars($row['sku_name']) ?></td>
                        <td class="text-center"><?= (int)$row['crates_loaded'] ?></td>
                        <td class="text-center">
                            <input type="number" name="returned[<?= $row['sku_id'] ?>][crates]" value="0" min="0" class="form-control crate-input text-center" 
                            data-loaded="<?= (int)$row['crates_loaded'] ?>"
                            data-loaves="<?= (int)$row['loaves_per_crate'] ?>"
                            data-price="<?= (float)$row['price_per_crate'] ?>">
                        </td>
                        <td class="text-center">
                            <input type="number" name="returned[<?= $row['sku_id'] ?>][pieces]" value="0" min="0" class="form-control piece-input text-center">
                        </td>
                        <td class="sold text-center">0</td>
                        <td class="val text-end">0.00</td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <div class="row g-3 mt-3 mb-4">
            <div class="col-md-4">
                <label class="form-label">Mileage End</label>
                <input type="number" name="mileage_end" class="form-control" required>
            </div>
            <div class="col-md-4">
                <label class="form-label">Distance Covered</label>
                <input type="text" id="distance" class="form-control" readonly>
            </div>
            <div class="col-md-4">
                <label class="form-label">Total Value Sold</label>
                <input type="text" id="total_value" class="form-control text-end" readonly>
            </div>
        </div>

        <div class="sticky-footer text-end">
            <button type="submit" class="btn btn-success">✅ Save Returns</button>
        </div>
    </form>
<?php endif; ?>

</div> <!-- End main-content -->

<script>
const crateInputs = document.querySelectorAll('.crate-input');
const pieceInputs = document.querySelectorAll('.piece-input');
const soldCells = document.querySelectorAll('.sold');
const valCells = document.querySelectorAll('.val');
const totalVal = document.getElementById('total_value');
const distance = document.getElementById('distance');
const startMileage = <?= $loading ? (int)$loading['mileage_start'] : 0 ?>;

function calculate() {
    let total = 0;
    crateInputs.forEach((crateInput, i) => {
        const pieceInput = pieceInputs[i];
        const loadedCrates = parseInt(crateInput.dataset.loaded);
        const loavesPerCrate = parseInt(crateInput.dataset.loaves);
        const pricePerCrate = parseFloat(crateInput.dataset.price);

        const returnedCrates = parseInt(crateInput.value) || 0;
        const returnedPieces = parseInt(pieceInput.value) || 0;

        const loadedLoaves = loadedCrates * loavesPerCrate;
        const returnedLoaves = (returnedCrates * loavesPerCrate) + returnedPieces;
        const soldLoaves = loadedLoaves - returnedLoaves;
        const soldCratesEquivalent = soldLoaves / loavesPerCrate;
        const valueSold = soldCratesEquivalent * pricePerCrate;

        soldCells[i].textContent = soldLoaves + " pcs";
        valCells[i].textContent = valueSold.toFixed(2);
        total += valueSold;
    });

    totalVal.value = total.toFixed(2);

    const endMileage = parseInt(document.querySelector("input[name='mileage_end']").value) || 0;
    if (endMileage > 0 && startMileage > 0) {
        distance.value = (endMileage - startMileage) + " km";
    }
}

crateInputs.forEach(input => input.addEventListener('input', calculate));
pieceInputs.forEach(input => input.addEventListener('input', calculate));
document.querySelector("input[name='mileage_end']").addEventListener('input', calculate);
</script>

</body>
</html>
