<?php
session_start();
require_once '../../config/db.php';

// Enable error logging
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

// Log POST data
file_put_contents('debug.txt', "POST Data: " . print_r($_POST, true) . "\n", FILE_APPEND);

// Validate request method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error'] = "Invalid request method.";
    header("Location: ../bread/returns.php");
    exit;
}

$loading_id = (int) ($_POST['loading_id'] ?? 0);
$cash = floatval($_POST['cash'] ?? 0);
$cheque = floatval($_POST['cheque'] ?? 0);
$cheque_no = trim($_POST['cheque_no'] ?? '');
$remarks = trim($_POST['remarks'] ?? '');

$mpesa_array = $_POST['mpesa'] ?? [];
$total_mpesa = 0;
foreach ($mpesa_array as $val) {
    $total_mpesa += floatval($val);
}

$total_payment = $cash + $cheque + $total_mpesa;

// Determine payment_mode
$payment_mode = 'cash';
if ($total_mpesa > 0 && $cash == 0 && $cheque == 0) {
    $payment_mode = 'mpesa';
} elseif ($cheque > 0 && $cash == 0 && $total_mpesa == 0) {
    $payment_mode = 'cheque';
} elseif ($cash > 0 && ($total_mpesa > 0 || $cheque > 0)) {
    $payment_mode = 'mixed';
}

// Validate loading ID
if ($loading_id <= 0) {
    $_SESSION['error'] = "Invalid loading ID.";
    header("Location: ../bread/returns.php");
    exit;
}

if ($total_payment <= 0) {
    $_SESSION['error'] = "No payment amount entered.";
    header("Location: ../bread/returns.php");
    exit;
}

// Confirm loading exists
$stmt = $conn->prepare("SELECT id FROM loadings WHERE id = ?");
$stmt->bind_param("i", $loading_id);
$stmt->execute();
if ($stmt->get_result()->num_rows === 0) {
    $_SESSION['error'] = "Loading session not found.";
    header("Location: ../bread/returns.php");
    exit;
}
$stmt->close();

$customer_id = null; // Optional: adjust if you associate a customer later

try {
    $conn->begin_transaction();

    // Insert into bread_payments
    $stmt = $conn->prepare("
        INSERT INTO bread_payments (
            loading_id, customer_id, payment_date,
            cash_amount, mpesa_amount, cheque_amount,
            cheque_no, payment_mode, remarks, created_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $conn->error);
    }

    $payment_date = date('Y-m-d');
    $stmt->bind_param(
        "iisdddsss",
        $loading_id,
        $customer_id,
        $payment_date,
        $cash,
        $total_mpesa,
        $cheque,
        $cheque_no,
        $payment_mode,
        $remarks
    );

    if (!$stmt->execute()) {
        throw new Exception("Payment insert failed: " . $stmt->error);
    }

    $payment_id = $conn->insert_id;
    $stmt->close();

    // Insert MPESA transactions
    if (!empty($mpesa_array)) {
        $stmt_mpesa = $conn->prepare("
            INSERT INTO mpesa_transactions (loading_id, payment_id, amount, created_at)
            VALUES (?, ?, ?, NOW())
        ");
        foreach ($mpesa_array as $mpesa_amt) {
            $amt = floatval($mpesa_amt);
            if ($amt > 0) {
                $stmt_mpesa->bind_param("iid", $loading_id, $payment_id, $amt);
                if (!$stmt_mpesa->execute()) {
                    throw new Exception("MPESA insert failed: " . $stmt_mpesa->error);
                }
            }
        }
        $stmt_mpesa->close();
    }

    $conn->commit();
    $_SESSION['success'] = "✅ Payment recorded successfully!";
    header("Location: print_receipt.php?id=" . $payment_id);
    exit;

} catch (Exception $e) {
    $conn->rollback();
    file_put_contents('debug.txt', "Error: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = "❌ Failed to record payment: " . $e->getMessage();
    header("Location: payments.php?loading_id=" . $loading_id);
    exit;
}
?>
