<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

$loading_id = isset($_GET['loading_id']) ? (int)$_GET['loading_id'] : 0;
if ($loading_id <= 0) {
    echo "<div class='alert alert-danger'>Invalid loading ID.</div>";
    exit;
}

// Fetch loading session
$stmt = $conn->prepare("
    SELECT l.id, l.loading_date, l.mileage_start, l.mileage_end,
           v.plate_number AS vehicle_plate, v.driver_name,
           r.name AS route_name
    FROM loadings l
    JOIN vehicles v ON l.vehicle_id = v.id
    JOIN routes r ON l.route_id = r.id
    WHERE l.id = ?
");
$stmt->bind_param("i", $loading_id);
$stmt->execute();
$loading = $stmt->get_result()->fetch_assoc();
if (!$loading) {
    echo "<div class='alert alert-danger'>Loading session not found.</div>";
    exit;
}

// Fetch loaded items
$stmt_items = $conn->prepare("
    SELECT li.sku_id, s.name AS sku_name, li.crates_loaded, s.price_per_crate, s.loaves_per_crate
    FROM loading_items li
    JOIN skus s ON li.sku_id = s.id
    WHERE li.loading_id = ?
");
$stmt_items->bind_param("i", $loading_id);
$stmt_items->execute();
$result_items = $stmt_items->get_result();

$items = [];
while ($row = $result_items->fetch_assoc()) {
    $row['returned_crates'] = 0;
    $row['returned_pieces'] = 0;
    $items[$row['sku_id']] = $row;
}

// Fetch returns
$stmt_returns = $conn->prepare("SELECT sku_id, returned_crates, returned_pieces FROM returns WHERE loading_id = ?");
$stmt_returns->bind_param("i", $loading_id);
$stmt_returns->execute();
$result_returns = $stmt_returns->get_result();
while ($r = $result_returns->fetch_assoc()) {
    if (isset($items[$r['sku_id']])) {
        $items[$r['sku_id']]['returned_crates'] += $r['returned_crates'];
        $items[$r['sku_id']]['returned_pieces'] += $r['returned_pieces'];
    }
}

// Fetch payments
$stmt_pay = $conn->prepare("
    SELECT 
        IFNULL(SUM(cash_amount), 0) AS cash,
        IFNULL(SUM(mpesa_amount), 0) AS mpesa,
        IFNULL(SUM(cheque_amount), 0) AS cheque
    FROM bread_payments
    WHERE loading_id = ?
");
$stmt_pay->bind_param("i", $loading_id);
$stmt_pay->execute();
$pay_data = $stmt_pay->get_result()->fetch_assoc();

$total_paid = $pay_data['cash'] + $pay_data['mpesa'] + $pay_data['cheque'];
?>

<!DOCTYPE html>
<html>
<head>
    <title>View Returns & Payment Summary</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</head>
<body>
<div class="container py-4">
    <h4 class="mb-4 text-primary"><i class="fas fa-eye me-2"></i> Bread Summary - Loading #<?= $loading['id'] ?></h4>

    <table class="table table-bordered w-75">
        <tr><th>Loading Date</th><td><?= $loading['loading_date'] ?></td></tr>
        <tr><th>Vehicle</th><td><?= $loading['vehicle_plate'] ?> - <?= $loading['driver_name'] ?></td></tr>
        <tr><th>Route</th><td><?= $loading['route_name'] ?></td></tr>
        <tr><th>Mileage Start</th><td><?= $loading['mileage_start'] ?></td></tr>
        <tr><th>Mileage End</th><td><?= $loading['mileage_end'] ?></td></tr>
    </table>

    <h5 class="mt-4">📦 SKU Breakdown</h5>
    <table class="table table-bordered table-striped">
        <thead class="table-light">
            <tr>
                <th>SKU</th>
                <th class="text-center">Crates Loaded</th>
                <th class="text-center">Crates Returned</th>
                <th class="text-center">Pieces Returned</th>
                <th class="text-center">Sold (pcs)</th>
                <th class="text-end">Expected Amount (Ksh)</th>
            </tr>
        </thead>
        <tbody>
        <?php
        $total_expected = 0;
        foreach ($items as $item):
            $loaded = $item['crates_loaded'] * $item['loaves_per_crate'];
            $returned = ($item['returned_crates'] * $item['loaves_per_crate']) + $item['returned_pieces'];
            $sold = max(0, $loaded - $returned);
            $crate_equiv = $sold / $item['loaves_per_crate'];
            $expected = $crate_equiv * $item['price_per_crate'];
            $total_expected += $expected;
        ?>
        <tr>
            <td><?= htmlspecialchars($item['sku_name']) ?></td>
            <td class="text-center"><?= $item['crates_loaded'] ?></td>
            <td class="text-center"><?= $item['returned_crates'] ?></td>
            <td class="text-center"><?= $item['returned_pieces'] ?></td>
            <td class="text-center"><?= $sold ?></td>
            <td class="text-end"><?= number_format($expected, 2) ?></td>
        </tr>
        <?php endforeach; ?>
        </tbody>
        <tfoot>
        <tr class="table-light fw-bold">
            <td colspan="5" class="text-end">Total Expected</td>
            <td class="text-end"><?= number_format($total_expected, 2) ?></td>
        </tr>
        </tfoot>
    </table>

    <h5 class="mt-4">💰 Payment Summary</h5>
    <table class="table table-bordered w-50">
        <tr><th>Cash</th><td class="text-end"><?= number_format($pay_data['cash'], 2) ?></td></tr>
        <tr><th>MPESA</th><td class="text-end"><?= number_format($pay_data['mpesa'], 2) ?></td></tr>
        <tr><th>Cheque</th><td class="text-end"><?= number_format($pay_data['cheque'], 2) ?></td></tr>
        <tr><th>Total Paid</th><td class="text-end text-success fw-bold"><?= number_format($total_paid, 2) ?></td></tr>
        <tr><th>Outstanding</th><td class="text-end text-danger fw-bold" id="outstanding_display"><?= number_format($total_expected - $total_paid, 2) ?></td></tr>
    </table>

    <div class="text-center mt-4">
        <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#paymentModal">💵 Make Payment</button>
        <a href="returns.php" class="btn btn-secondary">← Back</a>
    </div>
</div>

<!-- Modal -->
<div class="modal fade" id="paymentModal" tabindex="-1">
  <div class="modal-dialog">
    <form method="POST" action="payments.php" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Record Payment</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" name="loading_id" value="<?= $loading_id ?>">

        <div class="mb-2">
          <label>Cash Amount</label>
          <input type="number" step="0.01" name="cash" class="form-control calc" value="0">
        </div>

        <label>MPESA Payments</label>
        <div id="mpesa-container">
          <div class="input-group mb-2">
            <input type="number" step="0.01" name="mpesa[]" class="form-control calc" placeholder="MPESA Amount">
            <button type="button" class="btn btn-outline-secondary add-mpesa">+</button>
          </div>
        </div>

        <div class="mb-2">
          <label>Cheque Amount</label>
          <input type="number" step="0.01" name="cheque" class="form-control calc" value="0">
        </div>
        <div class="mb-2">
          <label>Cheque Number</label>
          <input type="text" name="cheque_no" class="form-control">
        </div>
        <div class="mb-2">
          <label>Remarks</label>
          <textarea name="remarks" class="form-control"></textarea>
        </div>

        <div class="alert alert-light border mt-3">
          <strong>Total Entered: </strong> <span id="total_display">0.00</span><br>
          <strong>Outstanding: </strong> <span id="outstanding_calc"><?= number_format($total_expected - $total_paid, 2) ?></span>
        </div>
      </div>
      <div class="modal-footer">
        <button type="submit" class="btn btn-success">Save Payment</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
      </div>
    </form>
  </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
    const calcFields = () => {
        let total = 0;
        document.querySelectorAll('.calc').forEach(el => {
            total += parseFloat(el.value || 0);
        });
        document.getElementById('total_display').textContent = total.toFixed(2);
        const outstanding = <?= json_encode($total_expected - $total_paid) ?> - total;
        document.getElementById('outstanding_calc').textContent = outstanding.toFixed(2);
    };

    document.body.addEventListener('input', e => {
        if (e.target.classList.contains('calc')) calcFields();
    });

    document.querySelector('.add-mpesa').addEventListener('click', () => {
        const container = document.getElementById('mpesa-container');
        const div = document.createElement('div');
        div.className = 'input-group mb-2';
        div.innerHTML = `
            <input type="number" step="0.01" name="mpesa[]" class="form-control calc" placeholder="MPESA Amount">
            <button type="button" class="btn btn-outline-danger remove-mpesa">−</button>`;
        container.appendChild(div);
    });

    document.body.addEventListener('click', e => {
        if (e.target.classList.contains('remove-mpesa')) {
            e.target.closest('.input-group').remove();
            calcFields();
        }
    });
});
</script>

</body>
</html>
