<?php
session_start();
require_once '../../config/db.php';


// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Fetch branches, suppliers, invoices, and available cheque leaves
$branches_query = "SELECT id, branch_name FROM branches ORDER BY branch_name";
$suppliers_query = "SELECT id, name FROM suppliers ORDER BY name"; // Changed 'supplier_name' to 'name'
$invoices_query = "SELECT id, invoice_number FROM invoices WHERE status = 'pending' ORDER BY invoice_number";
$leaves_query = "SELECT cl.id, cl.serial_number, cb.bank_name 
                FROM cheque_leaves cl 
                JOIN cheque_books cb ON cl.cheque_book_id = cb.id 
                WHERE cl.status = 'available' 
                ORDER BY cb.bank_name, cl.serial_number";

$branches = $conn->query($branches_query)->fetch_all(MYSQLI_ASSOC);
$suppliers = $conn->query($suppliers_query)->fetch_all(MYSQLI_ASSOC);
$invoices = $conn->query($invoices_query)->fetch_all(MYSQLI_ASSOC);
$cheque_leaves = $conn->query($leaves_query)->fetch_all(MYSQLI_ASSOC);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $branch_id = (int)$_POST['branch_id'];
    $supplier_id = !empty($_POST['supplier_id']) ? (int)$_POST['supplier_id'] : null;
    $invoice_id = !empty($_POST['invoice_id']) ? (int)$_POST['invoice_id'] : null;
    $cheque_leaf_id = (int)$_POST['cheque_leaf_id'];
    $cheque_number = trim($_POST['cheque_number']);
    $payee_name = trim($_POST['payee_name']);
    $amount = (float)$_POST['amount'];
    $issue_date = $_POST['issue_date'];
    $bank_name = trim($_POST['bank_name']);
    $status = $_POST['status'];
    $description = trim($_POST['description']);
   

    $errors = [];
    if (empty($branch_id)) $errors[] = "Branch is required.";
    if (empty($cheque_leaf_id)) $errors[] = "Cheque leaf is required.";
    if (empty($cheque_number)) $errors[] = "Cheque number is required.";
    if (empty($payee_name)) $errors[] = "Payee name is required.";
    if ($amount <= 0) $errors[] = "Amount must be greater than zero.";
    if (empty($issue_date)) $errors[] = "Issue date is required.";
    if (empty($bank_name)) $errors[] = "Bank name is required.";
    if (!in_array($status, ['issued', 'cleared', 'cancelled', 'returned'])) $errors[] = "Invalid status.";

    // Verify cheque leaf availability
    $leaf_check = "SELECT status FROM cheque_leaves WHERE id = ? AND status = 'available'";
    $stmt = $conn->prepare($leaf_check);
    $stmt->bind_param('i', $cheque_leaf_id);
    $stmt->execute();
    if ($stmt->get_result()->num_rows === 0) {
        $errors[] = "Selected cheque leaf is not available.";
    }

    // Check for duplicate cheque number
    $check_query = "SELECT id FROM cheques WHERE cheque_number = ?";
    $stmt = $conn->prepare($check_query);
    $stmt->bind_param('s', $cheque_number);
    $stmt->execute();
    if ($stmt->get_result()->num_rows > 0) {
        $errors[] = "Cheque number already exists.";
    }

    if (empty($errors)) {
        $conn->begin_transaction();
        try {
            // Insert cheque
            $insert_query = "INSERT INTO cheques (branch_id, supplier_id, invoice_id, cheque_leaf_id, cheque_number, payee_name, amount, issue_date, bank_name, status, description, created_by)
                             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $conn->prepare($insert_query);
            $stmt->bind_param('iiiissdssssi', $branch_id, $supplier_id, $invoice_id, $cheque_leaf_id, $cheque_number, $payee_name, $amount, $issue_date, $bank_name, $status, $description, $created_by);
            $stmt->execute();

            // Update cheque leaf status
            $update_leaf = "UPDATE cheque_leaves SET status = 'used' WHERE id = ?";
            $stmt = $conn->prepare($update_leaf);
            $stmt->bind_param('i', $cheque_leaf_id);
            $stmt->execute();

            $conn->commit();
            header("Location: cheque_writing.php?success=Cheque created successfully");
            exit;
        } catch (Exception $e) {
            $conn->rollback();
            $errors[] = "Failed to create cheque: " . $e->getMessage();
        }
    }
}
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Create Cheque - Delta Supermarkets</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css" rel="stylesheet">
  <style>
    body { background: #f4f6f8; font-size: 13px; }
    .header {
      background: #007bff;
      color: #fff;
      padding: 15px;
      border-radius: 4px;
      margin-bottom: 20px;
      border-bottom: 2px solid #0056b3;
    }
    .btn-primary {
      background-color: #007bff;
      border-color: #007bff;
    }
    .btn-primary:hover {
      background-color: #0056b3;
      border-color: #0056b3;
    }
    .form-control, .form-select {
      font-size: 13px;
      padding: 4px 8px;
    }
  </style>
</head>
<body>
<main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
  <div class="container my-4">
    <div class="header text-center">
      <h4 class="fw-bold">💸 DELTA SUPERMARKETS - CREATE CHEQUE</h4>
    </div>

    <?php if (!empty($errors)): ?>
      <div class="alert alert-danger">
        <?php foreach ($errors as $error): ?>
          <p><?php echo htmlspecialchars($error); ?></p>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>

    <form method="POST" class="row g-3">
      <div class="col-md-4">
        <label class="form-label">Branch:</label>
        <select name="branch_id" class="form-select" required>
          <option value="">Select Branch</option>
          <?php foreach ($branches as $branch): ?>
            <option value="<?= $branch['id'] ?>"><?= htmlspecialchars($branch['branch_name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-md-4">
        <label class="form-label">Supplier (Optional):</label>
        <select name="supplier_id" class="form-select">
          <option value="">Select Supplier</option>
          <?php foreach ($suppliers as $supplier): ?>
            <option value="<?= $supplier['id'] ?>"><?= htmlspecialchars($supplier['name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-md-4">
        <label class="form-label">Invoice (Optional):</label>
        <select name="invoice_id" class="form-select">
          <option value="">Select Invoice</option>
          <?php foreach ($invoices as $invoice): ?>
            <option value="<?= $invoice['id'] ?>"><?= htmlspecialchars($invoice['invoice_number']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-md-4">
        <label class="form-label">Cheque Leaf:</label>
        <select name="cheque_leaf_id" class="form-select" required>
          <option value="">Select Cheque Leaf</option>
          <?php foreach ($cheque_leaves as $leaf): ?>
            <option value="<?= $leaf['id'] ?>" data-bank="<?= htmlspecialchars($leaf['bank_name']) ?>">
              <?= htmlspecialchars($leaf['serial_number']) ?> (<?= htmlspecialchars($leaf['bank_name']) ?>)
            </option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-md-4">
        <label class="form-label">Cheque Number:</label>
        <input type="text" name="cheque_number" class="form-control" required>
      </div>
      <div class="col-md-4">
        <label class="form-label">Payee Name:</label>
        <input type="text" name="payee_name" class="form-control" required>
      </div>
      <div class="col-md-4">
        <label class="form-label">Amount (Ksh):</label>
        <input type="number" name="amount" class="form-control" step="0.01" min="0" required>
      </div>
      <div class="col-md-4">
        <label class="form-label">Issue Date:</label>
        <input type="date" name="issue_date" class="form-control" required>
      </div>
      <div class="col-md-4">
        <label class="form-label">Bank Name:</label>
        <input type="text" name="bank_name" class="form-control" required>
      </div>
      <div class="col-md-4">
        <label class="form-label">Status:</label>
        <select name="status" class="form-select" required>
          <option value="issued">Issued</option>
          <option value="cleared">Cleared</option>
          <option value="cancelled">Cancelled</option>
          <option value="returned">Returned</option>
        </select>
      </div>
      <div class="col-md-12">
        <label class="form-label">Description (Optional):</label>
        <textarea name="description" class="form-control" rows="4"></textarea>
      </div>
      <div class="col-md-12">
        <button type="submit" class="btn btn-primary">Create Cheque</button>
        <a href="cheque_writing.php" class="btn btn-secondary">Cancel</a>
      </div>
    </form>
  </div>
</main>

<script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
$(document).ready(function() {
  $('select[name="cheque_leaf_id"]').change(function() {
    const bankName = $(this).find(':selected').data('bank');
    $('input[name="bank_name"]').val(bankName);
    $('input[name="cheque_number"]').val($(this).find(':selected').text().split(' ')[0]);
  });
});
</script>
</body>
</html>