<?php
require_once 'pdo_db.php';
session_start();

// Generate CSRF token if not set
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Handle layout ID and database queries
$layout_id = isset($_GET['layout_id']) ? intval($_GET['layout_id']) : 0;
$layout = null;
$fields = [];

if ($layout_id > 0) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM cheque_layouts WHERE id = ?");
        $stmt->execute([$layout_id]);
        $layout = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($layout) {
            $stmt2 = $pdo->prepare("SELECT * FROM layout_fields WHERE layout_id = ?");
            $stmt2->execute([$layout_id]);
            $fields = $stmt2->fetchAll(PDO::FETCH_ASSOC);
        } else {
            // Create a new layout if ID is invalid
            $stmt = $pdo->prepare("INSERT INTO cheque_layouts (date_format) VALUES (?)");
            $stmt->execute(['DD-MM-YYYY']);
            $layout_id = $pdo->lastInsertId();
            error_log("Created new layout with ID: $layout_id");
            header("Location: layout_editor.php?layout_id=$layout_id");
            exit;
        }
    } catch (PDOException $e) {
        error_log('Database error: ' . $e->getMessage());
        die('Database error: Please check server logs.');
    }
} else {
    // Create a new layout
    try {
        $stmt = $pdo->prepare("INSERT INTO cheque_layouts (date_format) VALUES (?)");
        $stmt->execute(['DD-MM-YYYY']);
        $layout_id = $pdo->lastInsertId();
        error_log("Created new layout with ID: $layout_id");
        header("Location: layout_editor.php?layout_id=$layout_id");
        exit;
    } catch (PDOException $e) {
        error_log('Database error: ' . $e->getMessage());
        die('Database error: Please check server logs.');
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <title>Cheque Layout Editor</title>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://code.jquery.com/ui/1.12.1/themes/base/jquery-ui.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.5.12/cropper.min.css">
    <style>
        #editor {
            position: relative;
            width: 800px;
            height: 350px;
            border: 1px solid #ccc;
            background-size: cover;
            overflow: hidden;
            margin: 20px auto;
        }
        .field {
            position: absolute;
            border: 1px dashed #333;
            padding: 2px 5px;
            cursor: move;
            background: rgba(255,255,255,0.6);
            min-width: 20px;
            min-height: 20px;
        }
        .field[contenteditable]:focus {
            outline: 2px solid #007bff;
        }
        .control-panel {
            max-width: 800px;
            margin: auto;
        }
        .measurement-info {
            position: fixed;
            bottom: 10px;
            right: 10px;
            background: #fff;
            padding: 10px;
            border: 1px solid #ccc;
            display: none;
            z-index: 1000;
        }
        .cropper-container {
            max-width: 100%;
        }
        .alert-dismissible {
            margin-top: 10px;
        }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4">
        <h3>WYSIWYG Cheque Layout Creator</h3>

        <!-- Upload Form -->
        <form id="uploadForm" method="POST" enctype="multipart/form-data" class="mb-3">
            <label class="form-label">Upload Cheque Background (PNG, JPEG, max 1MB):</label>
            <input type="file" name="background" id="backgroundInput" class="form-control" accept="image/png,image/jpeg" required>
            <input type="hidden" name="layout_id" value="<?= $layout_id ?>">
            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token']) ?>">
            <button type="submit" class="btn btn-primary mt-2">Upload</button>
        </form>

        <?php if ($layout): ?>
            <!-- Editor Canvas -->
            <div id="editor" style="background-image: url('../../Uploads/cheque_backgrounds/<?= htmlspecialchars($layout['background_image'] ?? '') ?>');">
                <?php foreach ($fields as $f): ?>
                    <div class="field"
                        data-id="<?= $f['id'] ?>"
                        data-name="<?= htmlspecialchars($f['field_name']) ?>"
                        contenteditable="true"
                        style="
                            left: <?= $f['position_x'] ?>px;
                            top: <?= $f['position_y'] ?>px;
                            font-size: <?= $f['font_size'] ?>px;
                            font-family: <?= htmlspecialchars($f['font_family']) ?>;
                            width: <?= $f['width'] ?>px;
                            height: <?= $f['height'] ?>px;
                            display: <?= $f['is_visible'] ? 'block' : 'none' ?>;
                            color: <?= htmlspecialchars($f['font_color'] ?? 'black') ?>;
                            font-style: <?= $f['font_style'] === 'italic' ? 'italic' : 'normal' ?>;
                            font-weight: <?= $f['font_style'] === 'bold' ? '700' : 'normal' ?>;
                        ">
                        <?= ucfirst(htmlspecialchars($f['field_name'])) ?>
                    </div>
                <?php endforeach; ?>
            </div>

            <!-- Control Panel -->
            <div class="control-panel mt-4">
                <div class="row mb-3">
                    <div class="col-md-6">
                        <button class="btn btn-success" onclick="saveLayout()">💾 Save Layout</button>
                        <button class="btn btn-warning" onclick="resetLayout()">🔄 Reset</button>
                    </div>
                    <div class="col-md-6 text-end">
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addFieldModal">➕ Add Field</button>
                    </div>
                </div>

                <!-- Field Visibility and Date Format -->
                <div class="card">
                    <div class="card-header">Manage Fields</div>
                    <div class="card-body">
                        <ul class="list-group">
                            <?php foreach ($fields as $f): ?>
                                <li class="list-group-item d-flex justify-content-between align-items-center">
                                    <div class="form-check">
                                        <input type="checkbox" class="form-check-input" id="field_<?= $f['id'] ?>" data-id="<?= $f['id'] ?>" <?= $f['is_visible'] ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="field_<?= $f['id'] ?>"><?= ucfirst(htmlspecialchars($f['field_name'])) ?></label>
                                    </div>
                                    <button class="btn btn-sm btn-info" onclick="openFieldSettings('<?= $f['id'] ?>')">⚙️ Settings</button>
                                </li>
                            <?php endforeach; ?>
                            <li class="list-group-item">
                                <label class="form-label">Date Format:</label>
                                <select id="dateFormat" class="form-select">
                                    <option value="DD-MM-YYYY" <?= $layout['date_format'] === 'DD-MM-YYYY' ? 'selected' : '' ?>>DD-MM-YYYY</option>
                                    <option value="MM-DD-YYYY" <?= $layout['date_format'] === 'MM-DD-YYYY' ? 'selected' : '' ?>>MM-DD-YYYY</option>
                                    <option value="YYYY-MM-DD" <?= $layout['date_format'] === 'YYYY-MM-DD' ? 'selected' : '' ?>>YYYY-MM-DD</option>
                                    <option value="MMM DD, YYYY" <?= $layout['date_format'] === 'MMM DD, YYYY' ? 'selected' : '' ?>>MMM DD, YYYY</option>
                                    <option value="DD MMM YYYY" <?= $layout['date_format'] === 'DD MMM YYYY' ? 'selected' : '' ?>>DD MMM YYYY</option>
                                </select>
                            </li>
                        </ul>
                    </div>
                </div>

                <!-- Instructions -->
                <p class="text-muted mt-2">Drag fields to adjust positions. Click to edit text. Use settings to adjust font, size, and color. Upload a cheque image for accurate alignment.</p>
            </div>

            <!-- Measurement Info -->
            <div class="measurement-info" id="measurementInfo"></div>
        <?php endif; ?>
    </div>

    <!-- Add Field Modal -->
    <div class="modal fade" id="addFieldModal" tabindex="-1" aria-labelledby="addFieldModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addFieldModalLabel">Add New Field</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="addFieldForm">
                        <div class="mb-3">
                            <label for="fieldName" class="form-label">Field Name:</label>
                            <input type="text" id="fieldName" class="form-control" required>
                        </div>
                        <button type="submit" class="btn btn-primary">Add Field</button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Field Settings Modal -->
    <div class="modal fade" id="fieldSettingsModal" tabindex="-1" aria-labelledby="fieldSettingsModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="fieldSettingsModalLabel">Field Settings</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="fieldSettingsForm">
                        <input type="hidden" id="fieldId">
                        <div class="mb-3">
                            <label for="fontSize" class="form-label">Font Size:</label>
                            <select id="fontSize" class="form-select">
                                <?php for ($i = 8; $i <= 32; $i++): ?>
                                    <option value="<?= $i ?>px"><?= $i ?>px</option>
                                <?php endfor; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="fontFamily" class="form-label">Font Family:</label>
                            <select id="fontFamily" class="form-select">
                                <option value="Arial">Arial</option>
                                <option value="Verdana">Verdana</option>
                                <option value="Courier">Courier</option>
                                <option value="Times">Times</option>
                                <option value="Calibri">Calibri</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="fontStyle" class="form-label">Font Style:</label>
                            <select id="fontStyle" class="form-select">
                                <option value="normal">Normal</option>
                                <option value="bold">Bold</option>
                                <option value="italic">Italic</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="fontColor" class="form-label">Font Color:</label>
                            <select id="fontColor" class="form-select">
                                <option value="black">Black</option>
                                <option value="red">Red</option>
                                <option value="blue">Blue</option>
                                <option value="green">Green</option>
                            </select>
                        </div>
                        <button type="submit" class="btn btn-primary">Apply</button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Crop Image Modal -->
    <div class="modal fade" id="cropImageModal" tabindex="-1" aria-labelledby="cropImageModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="cropImageModalLabel">Crop Cheque Image</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="img-container">
                        <img id="imageToCrop" src="" alt="Cheque Image">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="cropButton">Crop and Apply</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://code.jquery.com/ui/1.12.1/jquery-ui.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.5.12/cropper.min.js"></script>
    <script>
        let cropper;
        const editor = document.getElementById('editor');
        const toMM = px => Math.round(px * 0.264583);
        const toPX = mm => Math.round(mm / 0.264583);

        // Initialize Draggable and Resizable Fields
        $('.field').draggable({
            containment: '#editor',
            drag: function(event, ui) {
                showMeasurementInfo($(this), ui.position.top, ui.position.left, $(this).width(), $(this).height());
            },
            stop: function() {
                hideMeasurementInfo();
            }
        }).resizable({
            containment: '#editor',
            handles: 'e',
            resize: function(event, ui) {
                showMeasurementInfo($(this), ui.position.top, ui.position.left, ui.size.width, ui.size.height);
            },
            stop: function() {
                hideMeasurementInfo();
            }
        }).on('click', function() {
            openFieldSettings($(this).data('id'));
        });

        // Field Visibility Toggle
        $('.form-check-input').on('change', function() {
            const fieldId = $(this).data('id');
            $(`.field[data-id="${fieldId}"]`).css('display', this.checked ? 'block' : 'none');
        });

        // Add New Field
        $('#addFieldForm').on('submit', function(e) {
            e.preventDefault();
            const fieldName = $('#fieldName').val().trim();
            if (!fieldName) {
                alert('Field name is required.');
                return;
            }

            const newField = $(`
                <div class="field" data-id="new_${Date.now()}" data-name="${fieldName}" contenteditable="true"
                    style="left: 10px; top: 10px; font-size: 16px; font-family: Arial; width: 100px; height: 20px;">
                    ${fieldName.charAt(0).toUpperCase() + fieldName.slice(1)}
                </div>
            `).appendTo('#editor');

            newField.draggable({
                containment: '#editor',
                drag: function(event, ui) {
                    showMeasurementInfo($(this), ui.position.top, ui.position.left, $(this).width(), $(this).height());
                },
                stop: function() {
                    hideMeasurementInfo();
                }
            }).resizable({
                containment: '#editor',
                handles: 'e',
                resize: function(event, ui) {
                    showMeasurementInfo($(this), ui.position.top, ui.position.left, ui.size.width, ui.size.height);
                },
                stop: function() {
                    hideMeasurementInfo();
                }
            }).on('click', function() {
                openFieldSettings($(this).data('id'));
            });

            $('#addFieldModal').modal('hide');
            $('#addFieldForm')[0].reset();
        });

        // Open Field Settings
        function openFieldSettings(fieldId) {
            const field = $(`.field[data-id="${fieldId}"]`);
            $('#fieldId').val(fieldId);
            $('#fontSize').val(field.css('font-size'));
            $('#fontFamily').val(field.css('font-family').replace(/['"]/g, ''));
            $('#fontStyle').val(field.css('font-style') === 'italic' ? 'italic' : field.css('font-weight') === '700' ? 'bold' : 'normal');
            $('#fontColor').val(field.css('color').toLowerCase() === 'rgb(255, 0, 0)' ? 'red' : field.css('color').toLowerCase() === 'rgb(0, 0, 255)' ? 'blue' : field.css('color').toLowerCase() === 'rgb(0, 128, 0)' ? 'green' : 'black');
            $('#fieldSettingsModal').modal('show');
        }

        // Apply Field Settings
        $('#fieldSettingsForm').on('submit', function(e) {
            e.preventDefault();
            const fieldId = $('#fieldId').val();
            const field = $(`.field[data-id="${fieldId}"]`);
            field.css({
                'font-size': $('#fontSize').val(),
                'font-family': $('#fontFamily').val(),
                'font-style': $('#fontStyle').val() === 'italic' ? 'italic' : 'normal',
                'font-weight': $('#fontStyle').val() === 'bold' ? '700' : 'normal',
                'color': $('#fontColor').val()
            });
            $('#fieldSettingsModal').modal('hide');
        });

        // Image Upload and Cropping
        $('#uploadForm').on('submit', function(e) {
            e.preventDefault();
            const file = $('#backgroundInput')[0].files[0];
            const layoutId = $('input[name="layout_id"]').val();
            console.log('Submitting form with layout_id:', layoutId);
            if (!file) {
                alert('Please select an image.');
                return;
            }
            if (file.size > 1048576) {
                alert('Image size must be less than 1MB.');
                return;
            }
            if (!['image/png', 'image/jpeg'].includes(file.type)) {
                alert('Only PNG and JPEG images are allowed.');
                return;
            }
            if (!layoutId || layoutId <= 0) {
                alert('Invalid layout ID. Please reload the page and try again.');
                return;
            }

            const reader = new FileReader();
            reader.onload = function(e) {
                $('#imageToCrop').attr('src', e.target.result);
                $('#cropImageModal').modal('show');
                setTimeout(() => {
                    try {
                        if (cropper) cropper.destroy();
                        cropper = new Cropper(document.getElementById('imageToCrop'), {
                            aspectRatio: 800 / 350,
                            viewMode: 1,
                            autoCropArea: 1,
                            movable: false,
                            zoomable: false,
                            ready() {
                                console.log('Cropper initialized successfully');
                            }
                        });
                    } catch (err) {
                        console.error('Cropper initialization failed:', err);
                        alert('Failed to initialize image cropper. Please try again.');
                        $('#cropImageModal').modal('hide');
                    }
                }, 500);
            };
            reader.onerror = function() {
                alert('Error reading the image file.');
            };
            reader.readAsDataURL(file);
        });

        $('#cropButton').on('click', function() {
            if (!cropper) {
                alert('Cropper not initialized. Please try again.');
                return;
            }
            try {
                const canvas = cropper.getCroppedCanvas({ width: 800, height: 350 });
                if (!canvas) {
                    throw new Error('Failed to generate cropped canvas');
                }
                canvas.toBlob(blob => {
                    if (!blob) {
                        alert('Failed to generate image blob.');
                        return;
                    }
                    const formData = new FormData();
                    formData.append('background', blob, 'cheque_background.jpg');
                    formData.append('layout_id', <?= $layout_id ?>);
                    formData.append('csrf_token', '<?= htmlspecialchars($_SESSION['csrf_token']) ?>');

                    console.log('Sending formData with layout_id:', <?= $layout_id ?>);
                    fetch('upload_bg.php', {
                        method: 'POST',
                        body: formData
                    })
                    .then(res => {
                        if (!res.ok) {
                            throw new Error(`HTTP error! Status: ${res.status}`);
                        }
                        return res.json();
                    })
                    .then(data => {
                        if (data.success) {
                            $('#editor').css('background-image', `url('../../Uploads/cheque_backgrounds/${data.filename}')`);
                            $('#cropImageModal').modal('hide');
                            alert('Image uploaded and cropped successfully!');
                        } else {
                            console.error('Upload error:', data.error);
                            alert(`Error uploading image: ${data.error}`);
                        }
                    })
                    .catch(err) {
                        console.error('Fetch error:', err);
                        alert('Failed to upload image. Please check the console for details.');
                    });
                }, 'image/jpeg', 0.8);
            } catch (err) {
                console.error('Cropping error:', err);
                alert('Error processing the cropped image.');
            }
            cropper.destroy();
            cropper = null;
        });

        // Save Layout
        function saveLayout() {
            const fields = [];
            $('.field').each(function() {
                fields.push({
                    id: $(this).data('id'),
                    name: $(this).data('name'),
                    x: parseInt($(this).css('left')) || 0,
                    y: parseInt($(this).css('top')) || 0,
                    font_size: parseInt($(this).css('font-size')) || 16,
                    font_family: $(this).css('font-family').replace(/['"]/g, '') || 'Arial',
                    font_style: $(this).css('font-style') === 'italic' ? 'italic' : $(this).css('font-weight') === '700' ? 'bold' : 'normal',
                    font_color: $(this).css('color').toLowerCase() === 'rgb(255, 0, 0)' ? 'red' : $(this).css('color').toLowerCase() === 'rgb(0, 0, 255)' ? 'blue' : $(this).css('color').toLowerCase() === 'rgb(0, 128, 0)' ? 'green' : 'black',
                    width: parseInt($(this).css('width')) || 100,
                    height: parseInt($(this).css('height')) || 20,
                    visible: $(this).css('display') !== 'none' ? 1 : 0
                });
            });

            fetch('save_layout.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    layout_id: <?= $layout_id ?>,
                    fields,
                    date_format: $('#dateFormat').val(),
                    csrf_token: '<?= htmlspecialchars($_SESSION['csrf_token']) ?>'
                })
            })
            .then(res => {
                if (!res.ok) {
                    throw new Error(`HTTP error! Status: ${res.status}`);
                }
                return res.json();
            })
            .then(data => {
                if (data.success) {
                    alert('Layout saved successfully!');
                } else {
                    console.error('Save error:', data.error);
                    alert(`Error saving layout: ${data.error}`);
                }
            })
            .catch(err) {
                console.error('Fetch error:', err);
                alert('Failed to save layout. Please check the console for details.');
            });
        }

        // Reset Layout
        function resetLayout() {
            if (confirm('Reset positions to original?')) location.reload();
        }

        // Show Measurement Info
        function showMeasurementInfo(el, top, left, width, height) {
            const info = `
                <div><strong>${el.data('name')}</strong></div>
                <div>Left: ${toMM(left)}mm</div>
                <div>Top: ${toMM(top)}mm</div>
                <div>Width: ${toMM(width)}mm</div>
                <div>Height: ${toMM(height)}mm</div>
            `;
            $('#measurementInfo').html(info).show();
        }

        // Hide Measurement Info
        function hideMeasurementInfo() {
            $('#measurementInfo').fadeOut('fast');
        }
    </script>
</body>
</html>