<?php
session_start();
require_once '../../config/db.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

$vehicle_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
if ($vehicle_id <= 0) {
    $_SESSION['error'] = "Invalid vehicle selected.";
    header("Location: list_vehicles.php");
    exit();
}

// Fetch vehicle + driver + branch
$stmt = $conn->prepare("
    SELECT v.*, b.branch_name, d.full_name AS driver_name, d.phone AS driver_phone
    FROM vehicles v
    LEFT JOIN branches b ON v.branch_id = b.id
    LEFT JOIN drivers d ON v.assigned_driver_id = d.id
    WHERE v.id = ?
");
$stmt->bind_param("i", $vehicle_id);
$stmt->execute();
$vehicle = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$vehicle) {
    $_SESSION['error'] = "Vehicle not found.";
    header("Location: list_vehicles.php");
    exit();
}
include_once '../../includes/layout2.php';

?>

<div class="main-content p-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="fw-bold text-primary">
            <i class="fas fa-eye me-2"></i> Vehicle Profile - <?= htmlspecialchars($vehicle['plate_number']) ?>
        </h4>
        <a href="list_vehicles.php" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back</a>
    </div>

    <!-- Vehicle Overview -->
    <div class="card shadow-sm mb-4">
        <div class="card-body">
            <div class="row g-3">
                <div class="col-md-4"><strong>Plate Number:</strong> <?= htmlspecialchars($vehicle['plate_number']) ?></div>
                <div class="col-md-4"><strong>Make:</strong> <?= htmlspecialchars($vehicle['make']) ?></div>
                <div class="col-md-4"><strong>Model:</strong> <?= htmlspecialchars($vehicle['model']) ?></div>
                <div class="col-md-3"><strong>Year:</strong> <?= $vehicle['year'] ?></div>
                <div class="col-md-3"><strong>Fuel:</strong> <?= $vehicle['fuel_type'] ?></div>
                <div class="col-md-3"><strong>Tank Capacity:</strong> <?= $vehicle['tank_capacity'] ?> L</div>
                <div class="col-md-3"><strong>Odometer:</strong> <?= $vehicle['odometer'] ?> km</div>
                <div class="col-md-4"><strong>Status:</strong>
                    <span class="badge bg-<?= match($vehicle['status']) {
                        'Active' => 'success',
                        'Maintenance' => 'warning',
                        default => 'secondary'
                    } ?>">
                        <?= $vehicle['status'] ?>
                    </span>
                </div>
                <div class="col-md-4"><strong>Branch:</strong> <?= htmlspecialchars($vehicle['branch_name']) ?></div>
                <div class="col-md-4"><strong>Created:</strong> <?= $vehicle['created_at'] ?></div>
            </div>
        </div>
    </div>

    <!-- Tabbed Section -->
    <ul class="nav nav-tabs mb-3" id="vehicleTabs" role="tablist">
        <li class="nav-item"><a class="nav-link active" data-bs-toggle="tab" href="#driver">👤 Driver</a></li>
        <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#trips">🛣 Trips</a></li>
        <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#fuel">⛽ Fuel</a></li>
        <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#maintenance">🔧 Maintenance</a></li>
        <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#docs">📎 Documents</a></li>
    </ul>

    <div class="tab-content" id="vehicleTabContent">
        <!-- DRIVER -->
        <div class="tab-pane fade show active" id="driver">
            <div class="card card-body">
                <?php if ($vehicle['driver_name']): ?>
                    <p><strong>Name:</strong> <?= htmlspecialchars($vehicle['driver_name']) ?></p>
                    <p><strong>Phone:</strong> <?= htmlspecialchars($vehicle['driver_phone']) ?></p>
                <?php else: ?>
                    <p class="text-muted">No driver assigned.</p>
                <?php endif; ?>
                <a href="assign_driver.php?vehicle_id=<?= $vehicle['id'] ?>" class="btn btn-outline-primary btn-sm">
                    <i class="fas fa-user-tag"></i> Assign/Change Driver
                </a>
            </div>
        </div>

        <!-- TRIPS -->
        <div class="tab-pane fade" id="trips">
            <div class="card card-body">
                <?php
                $res = $conn->query("SELECT * FROM fleet_trips WHERE vehicle_id = $vehicle_id ORDER BY trip_date DESC");
                if ($res && $res->num_rows > 0): ?>
                    <table class="table table-sm table-bordered">
                        <thead><tr>
                            <th>Date</th><th>Origin</th><th>Destination</th><th>Distance (km)</th><th>Purpose</th>
                        </tr></thead>
                        <tbody>
                            <?php while ($trip = $res->fetch_assoc()): ?>
                                <tr>
                                    <td><?= $trip['trip_date'] ?></td>
                                    <td><?= htmlspecialchars($trip['origin']) ?></td>
                                    <td><?= htmlspecialchars($trip['destination']) ?></td>
                                    <td><?= number_format($trip['distance_km'], 1) ?></td>
                                    <td><?= htmlspecialchars($trip['purpose']) ?></td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p class="text-muted">No trip logs available.</p>
                <?php endif; ?>
            </div>
        </div>

        <!-- FUEL -->
        <div class="tab-pane fade" id="fuel">
            <div class="card card-body">
                <?php
                $res = $conn->query("SELECT * FROM fuel_logs WHERE vehicle_id = $vehicle_id ORDER BY refuel_date DESC");
                if ($res && $res->num_rows > 0): ?>
                    <table class="table table-sm table-bordered">
                        <thead><tr>
                            <th>Date</th><th>Liters</th><th>Cost</th><th>Station</th><th>Odometer</th>
                        </tr></thead>
                        <tbody>
                            <?php while ($fuel = $res->fetch_assoc()): ?>
                                <tr>
                                    <td><?= $fuel['refuel_date'] ?></td>
                                    <td><?= $fuel['liters'] ?></td>
                                    <td>KES <?= number_format($fuel['cost'], 2) ?></td>
                                    <td><?= htmlspecialchars($fuel['station_name']) ?></td>
                                    <td><?= $fuel['odometer'] ?> km</td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p class="text-muted">No fuel logs recorded.</p>
                <?php endif; ?>
            </div>
        </div>

        <!-- MAINTENANCE -->
        <div class="tab-pane fade" id="maintenance">
            <div class="card card-body">
                <?php
                $res = $conn->query("SELECT * FROM maintenance_logs WHERE vehicle_id = $vehicle_id ORDER BY service_date DESC");
                if ($res && $res->num_rows > 0): ?>
                    <table class="table table-sm table-bordered">
                        <thead><tr>
                            <th>Date</th><th>Type</th><th>Cost</th><th>Garage</th><th>Notes</th>
                        </tr></thead>
                        <tbody>
                            <?php while ($m = $res->fetch_assoc()): ?>
                                <tr>
                                    <td><?= $m['service_date'] ?></td>
                                    <td><?= htmlspecialchars($m['service_type']) ?></td>
                                    <td>KES <?= number_format($m['cost'], 2) ?></td>
                                    <td><?= htmlspecialchars($m['garage']) ?></td>
                                    <td><?= htmlspecialchars($m['notes']) ?></td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p class="text-muted">No maintenance records found.</p>
                <?php endif; ?>
            </div>
        </div>

        <!-- DOCUMENTS -->
        <div class="tab-pane fade" id="docs">
            <div class="card card-body">
                <?php
                $res = $conn->query("SELECT * FROM vehicle_documents WHERE vehicle_id = $vehicle_id ORDER BY uploaded_at DESC");
                if ($res && $res->num_rows > 0): ?>
                    <ul class="list-group">
                        <?php while ($doc = $res->fetch_assoc()): ?>
                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                <div>
                                    <strong><?= $doc['doc_type'] ?>:</strong> <?= basename($doc['file_path']) ?>
                                </div>
                                <a href="<?= htmlspecialchars($doc['file_path']) ?>" target="_blank" class="btn btn-sm btn-primary">
                                    <i class="fas fa-download"></i> View
                                </a>
                            </li>
                        <?php endwhile; ?>
                    </ul>
                <?php else: ?>
                    <p class="text-muted">No documents uploaded.</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php include_once '../../includes/footer.php'; ?>
