<?php
session_start();
require_once '../../config/db.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  $_SESSION['error'] = "Invalid request method.";
  header("Location: list.php");
  exit;
}

$action = $_POST['action'] ?? '';
$invoice_id = (int) ($_POST['invoice_id'] ?? 0);
$password = $_POST['password'] ?? '';
$user_id = $_SESSION['user_id'] ?? 0;
$user_role = $_SESSION['role'] ?? '';

if (!$action || !$invoice_id || !$password || !$user_id) {
  $_SESSION['error'] = "Missing required fields.";
  header("Location: list.php");
  exit;
}

// Fetch user's password hash
$stmt = $conn->prepare("SELECT password FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

if (!$result || $result->num_rows === 0) {
  $_SESSION['error'] = "User not found.";
  header("Location: list.php");
  exit;
}

$user = $result->fetch_assoc();
if (!password_verify($password, $user['password'])) {
  $_SESSION['error'] = "Invalid password. Action not authorized.";
  header("Location: view.php?id=$invoice_id");
  exit;
}

// Fetch invoice
$stmt = $conn->prepare("SELECT status FROM invoices WHERE id = ?");
$stmt->bind_param("i", $invoice_id);
$stmt->execute();
$res = $stmt->get_result();

if (!$res || $res->num_rows === 0) {
  $_SESSION['error'] = "Invoice not found.";
  header("Location: list.php");
  exit;
}

$invoice = $res->fetch_assoc();
$current_status = $invoice['status'];

// Define valid transitions
$transitions = [
  'submit'  => ['from' => 'Draft',     'to' => 'Submitted', 'roles' => ['Admin', 'Accountant']],
  'verify'  => ['from' => 'Submitted', 'to' => 'Verified',  'roles' => ['Admin', 'Manager']],
  'approve' => ['from' => 'Verified',  'to' => 'Approved',  'roles' => ['Admin', 'Manager']],
  'reject'  => ['from' => 'Submitted', 'to' => 'Rejected',  'roles' => ['Admin', 'Manager']],
  'cancel'  => ['from' => 'Draft',     'to' => 'Cancelled', 'roles' => ['Admin']],
];

if (!isset($transitions[$action])) {
  $_SESSION['error'] = "Invalid action requested.";
  header("Location: view.php?id=$invoice_id");
  exit;
}

$t = $transitions[$action];

// Role check
if (!in_array($user_role, $t['roles'])) {
  $_SESSION['error'] = "You are not authorized to perform this action.";
  header("Location: view.php?id=$invoice_id");
  exit;
}

// Status check
if ($current_status !== $t['from']) {
  $_SESSION['error'] = "Invoice must be in '{$t['from']}' status to {$action}.";
  header("Location: view.php?id=$invoice_id");
  exit;
}

// Build update query
$fields = "status = ?, updated_at = NOW(), updated_by = ?";
$params = [$t['to'], $user_id];
$types = "si";

if ($action === 'approve') {
  $fields .= ", approved_by = ?, approved_at = NOW()";
  $params[] = $user_id;
  $types .= "i";
}

$params[] = $invoice_id;
$types .= "i";

$update_sql = "UPDATE invoices SET $fields WHERE id = ?";
$update_stmt = $conn->prepare($update_sql);
$update_stmt->bind_param($types, ...$params);

if ($update_stmt->execute()) {
  $_SESSION['success'] = "✅ Invoice successfully marked as '{$t['to']}'.";

  // Log transition
  $log_stmt = $conn->prepare("
    INSERT INTO invoice_logs (invoice_id, user_id, action, timestamp)
    VALUES (?, ?, ?, NOW())
  ");
  $log_stmt->bind_param("iis", $invoice_id, $user_id, ucfirst($action));
  $log_stmt->execute();

} else {
  $_SESSION['error'] = "❌ Failed to update invoice status.";
}

header("Location: view.php?id=$invoice_id");
exit;
?>
