<?php
session_start();
require_once '../../config/db.php';
header('Content-Type: application/json');

$input = json_decode(file_get_contents('php://input'), true);
$merchandiser_id = (int)($input['merchandiser_id'] ?? 0);
$branch_id = (int)($input['branch_id'] ?? 0);
$admin_user = $_SESSION['username'] ?? 'System';

$block_cross_branch = true;  // 🚨 Set to true to block cross-branch logout

if (!$merchandiser_id || !$branch_id) {
    echo json_encode(['success' => false, 'message' => 'Invalid input']);
    exit;
}

// Get latest log
$stmt = $conn->prepare("SELECT log_type, branch_id FROM merchandiser_logs WHERE merchandiser_id = ? ORDER BY log_time DESC LIMIT 1");
$stmt->bind_param("i", $merchandiser_id);
$stmt->execute();
$last = $stmt->get_result()->fetch_assoc();

if (!$last || $last['log_type'] !== 'IN') {
    echo json_encode(['success' => false, 'message' => 'Merchandiser is not currently logged in']);
    exit;
}

if ($block_cross_branch && $last['branch_id'] != $branch_id) {
    // Audit log
    $msg = "Attempted cross-branch logout from branch $branch_id, but blocked (logged in at branch {$last['branch_id']}).";
    $log_stmt = $conn->prepare("INSERT INTO audit_logs (merchandiser_id, action, message, performed_by) VALUES (?, 'FORCE_LOGOUT_ATTEMPT', ?, ?)");
    $log_stmt->bind_param("iss", $merchandiser_id, $msg, $admin_user);
    $log_stmt->execute();

    echo json_encode(['success' => false, 'message' => 'Cross-branch logout is blocked by system settings.']);
    exit;
}

// Insert OUT log
$stmt = $conn->prepare("INSERT INTO merchandiser_logs (merchandiser_id, branch_id, log_type, log_time) VALUES (?, ?, 'OUT', NOW())");
$stmt->bind_param("ii", $merchandiser_id, $branch_id);
$stmt->execute();

// Audit log
$msg = ($last['branch_id'] != $branch_id)
    ? "Forced cross-branch logout from branch $branch_id (originally logged in at branch {$last['branch_id']})."
    : "Regular logout from same branch.";
$log_stmt = $conn->prepare("INSERT INTO audit_logs (merchandiser_id, action, message, performed_by) VALUES (?, 'FORCE_LOGOUT_SUCCESS', ?, ?)");
$log_stmt->bind_param("iss", $merchandiser_id, $msg, $admin_user);
$log_stmt->execute();

echo json_encode(['success' => true, 'message' => 'Force logout recorded']);
exit;
