<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// Initialize current month
$current_month = isset($_GET['month']) ? $_GET['month'] : date('Y-m');
$month_start = date('Y-m-01', strtotime($current_month));
$month_end = date('Y-m-t', strtotime($current_month));

// Fetch calendar events (transaction dates, counts, and payment totals)
$calendar_query = "
    SELECT 
        DATE(t.transaction_date) AS event_date,
        SUM(CASE WHEN t.transaction_type = 'invoice' THEN 1 ELSE 0 END) AS invoice_count,
        SUM(CASE WHEN t.transaction_type = 'payment' THEN 1 ELSE 0 END) AS payment_count,
        SUM(CASE WHEN t.transaction_type = 'payment' THEN t.amount ELSE 0 END) AS total_payment_amount
    FROM (
        SELECT i.invoice_date AS transaction_date, 'invoice' AS transaction_type, 0 AS amount
        FROM invoices i
        UNION ALL
        SELECT p.payment_date AS transaction_date, 'payment' AS transaction_type, p.amount AS amount
        FROM payments p
    ) t
    WHERE t.transaction_date BETWEEN ? AND ?
    GROUP BY DATE(t.transaction_date)
";

$calendar_stmt = $conn->prepare($calendar_query);
if ($calendar_stmt === false) {
    error_log("Calendar query preparation failed: " . $conn->error);
    die("Calendar query preparation failed: " . htmlspecialchars($conn->error));
}
$calendar_stmt->bind_param('ss', $month_start, $month_end);
if (!$calendar_stmt->execute()) {
    error_log("Calendar query execution failed: " . $calendar_stmt->error);
    die("Calendar query execution failed: " . htmlspecialchars($calendar_stmt->error));
}
$calendar_result = $calendar_stmt->get_result();
$events = [];
$total_invoices = 0;
$total_payments = 0;
while ($row = $calendar_result->fetch_assoc()) {
    $events[$row['event_date']] = [
        'invoices' => $row['invoice_count'],
        'payments' => $row['payment_count'],
        'total_payments' => $row['total_payment_amount']
    ];
    $total_invoices += $row['invoice_count'];
    $total_payments += $row['payment_count'];
}
$calendar_stmt->close();

// Cache events
if (!isset($_SESSION['calendar_events_' . $current_month])) {
    $_SESSION['calendar_events_' . $current_month] = $events;
    $_SESSION['calendar_totals_' . $current_month] = ['invoices' => $total_invoices, 'payments' => $total_payments];
} else {
    $events = $_SESSION['calendar_events_' . $current_month];
    $total_invoices = $_SESSION['calendar_totals_' . $current_month]['invoices'];
    $total_payments = $_SESSION['calendar_totals_' . $current_month]['payments'];
}

// Debug: Log schemas and query
$invoices_schema = $conn->query("DESCRIBE invoices");
$payments_schema = $conn->query("DESCRIBE payments");
if ($invoices_schema === false || $payments_schema === false) {
    error_log("Schema query failed: " . $conn->error);
}
error_log("Calendar Query: $calendar_query");
error_log("Params: month_start=$month_start, month_end=$month_end");
?>

<style>
    .calendar { width: 100%; margin: 0 auto; padding: 0 10px; }
    .calendar-header { 
        display: flex; 
        justify-content: space-between; 
        align-items: center; 
        margin-bottom: 20px; 
        background: linear-gradient(135deg, #4b6cb7, #182848); 
        padding: 15px; 
        border-radius: 8px; 
        color: #fff; 
    }
    .calendar-grid { 
        display: grid; 
        grid-template-columns: repeat(7, 1fr); 
        gap: 4px; 
        background-color: #e7f1ff; 
        border-radius: 8px; 
        padding: 4px; 
    }
    .calendar-day { 
        background-color: #fff; 
        padding: 15px; 
        text-align: center; 
        min-height: 160px; 
        position: relative; 
        font-size: 1.2em; 
        border-radius: 6px; 
        box-shadow: 0 2px 4px rgba(0,0,0,0.1); 
        transition: transform 0.2s, box-shadow 0.2s; 
    }
    .calendar-day.header { 
        background: #4b6cb7; 
        color: #fff; 
        font-weight: bold; 
        padding: 12px; 
        font-size: 1.1em; 
        box-shadow: none; 
    }
    .calendar-day.empty { 
        background-color: #f0f7ff; 
        box-shadow: none; 
    }
    .calendar-day.has-events { 
        cursor: pointer; 
    }
    .calendar-day.has-events:hover { 
        transform: scale(1.02); 
        box-shadow: 0 4px 8px rgba(0,0,0,0.2); 
        background-color: #f8fbff; 
    }
    .event-indicator { 
        position: absolute; 
        bottom: 10px; 
        left: 50%; 
        transform: translateX(-50%); 
        display: flex; 
        gap: 8px; 
    }
    .event-invoice { 
        width: 12px; 
        height: 12px; 
        background-color: #ff9999; 
        border-radius: 50%; 
        border: 1px solid #fff; 
    }
    .event-payment { 
        width: 12px; 
        height: 12px; 
        background-color: #99cc99; 
        border-radius: 50%; 
        border: 1px solid #fff; 
    }
    .event-summary { 
        font-size: 0.85em; 
        margin-top: 8px; 
        text-align: left; 
        line-height: 1.5; 
    }
    .event-summary span { 
        display: block; 
        font-weight: bold; 
        color: #343a40; 
    }
    .event-summary .payment-amount { 
        font-size: 1.2em; 
        color: #dc3545; 
        font-weight: bold; 
    }
    .summary { 
        text-align: center; 
        margin-bottom: 25px; 
        font-size: 1.4em; 
        font-weight: bold; 
        color: #343a40; 
        background: #f0f7ff; 
        padding: 15px; 
        border-radius: 8px; 
        box-shadow: 0 2px 4px rgba(0,0,0,0.1); 
    }
    .modal-header { 
        background: #4b6cb7; 
        color: #fff; 
    }
    .modal-content { 
        border: 1px solid #4b6cb7; 
    }
    .btn-close-white { 
        filter: invert(1); 
    }
    @media (max-width: 768px) {
        .calendar-day { 
            min-height: 120px; 
            font-size: 1em; 
            padding: 10px; 
        }
        .event-summary { font-size: 0.75em; }
        .event-summary .payment-amount { font-size: 1em; }
        .summary { font-size: 1.2em; padding: 10px; }
        .calendar-header { padding: 10px; }
    }
    @media (max-width: 576px) {
        .calendar-grid { grid-template-columns: repeat(7, 1fr); }
        .calendar-day { min-height: 100px; font-size: 0.9em; }
        .event-indicator { bottom: 8px; }
        .event-invoice, .event-payment { width: 10px; height: 10px; }
        .event-summary .payment-amount { font-size: 0.9em; }
    }
</style>

<div class="main-content">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="fw-bold text-primary"><i class="fas fa-calendar-alt me-2"></i> Transaction Calendar</h4>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success rounded-3"><?= $_SESSION['success']; unset($_SESSION['success']); ?></div>
    <?php elseif (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger rounded-3"><?= $_SESSION['error']; unset($_SESSION['error']); ?></div>
    <?php endif; ?>

    <!-- Summary of Contents -->
    <div class="summary" aria-label="Transaction summary for <?= date('F Y', strtotime($current_month)) ?>">
        <?= date('F Y', strtotime($current_month)) ?> Transactions: 
        <span class="text-danger"><?= $total_invoices ?> Invoices</span>, 
        <span class="text-success"><?= $total_payments ?> Payments</span>
    </div>

    <?php if (empty($events)): ?>
        <div class="alert alert-warning rounded-3">No transactions found for this month.</div>
    <?php endif; ?>

    <!-- Smart Calendar -->
    <div class="card shadow-sm mb-4 rounded-3" style="background: #f0f7ff;">
        <div class="card-body p-4">
            <div class="calendar">
                <div class="calendar-header">
                    <a href="?month=<?= date('Y-m', strtotime($current_month . ' -1 month')) ?>" class="btn btn-sm btn-outline-light">Previous</a>
                    <h5 class="mb-0 fw-bold text-white"><?= date('F Y', strtotime($current_month)) ?></h5>
                    <a href="?month=<?= date('Y-m', strtotime($current_month . ' +1 month')) ?>" class="btn btn-sm btn-outline-light">Next</a>
                </div>
                <div class="calendar-grid">
                    <?php
                    $days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
                    foreach ($days as $day) {
                        echo "<div class='calendar-day header'>$day</div>";
                    }
                    $first_day = date('w', strtotime($month_start));
                    $days_in_month = date('t', strtotime($current_month));
                    for ($i = 0; $i < $first_day; $i++) {
                        echo "<div class='calendar-day empty'></div>";
                    }
                    for ($day = 1; $day <= $days_in_month; $day++) {
                        $date = date('Y-m-d', strtotime("$current_month-$day"));
                        $has_events = isset($events[$date]);
                        $class = $has_events ? 'has-events' : '';
                        $tooltip = $has_events ? "Invoices: {$events[$date]['invoices']}, Payments: {$events[$date]['payments']}, Total Payments: KSh " . number_format($events[$date]['total_payments'], 2) : '';
                        echo "<div class='calendar-day $class' data-date='$date' role='button' aria-label='View transactions for $date' title='$tooltip'>";
                        echo $day;
                        if ($has_events) {
                            echo "<div class='event-summary'>";
                            if ($events[$date]['invoices'] > 0) {
                                echo "<span>Invoices: {$events[$date]['invoices']}</span>";
                            }
                            if ($events[$date]['payments'] > 0) {
                                echo "<span>Payments: {$events[$date]['payments']}</span>";
                                echo "<span class='payment-amount'>KSh " . number_format($events[$date]['total_payments'], 2) . "</span>";
                            }
                            echo "</div>";
                            echo "<div class='event-indicator'>";
                            if ($events[$date]['invoices'] > 0) {
                                echo "<span class='event-invoice'></span>";
                            }
                            if ($events[$date]['payments'] > 0) {
                                echo "<span class='event-payment'></span>";
                            }
                            echo "</div>";
                        }
                        echo "</div>";
                    }
                    ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal for Transaction Details -->
    <div class="modal fade" id="transactionModal" tabindex="-1" aria-labelledby="transactionModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content rounded-3">
                <div class="modal-header">
                    <h5 class="modal-title" id="transactionModalLabel">Transactions</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div id="transactionDetails"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary rounded-3" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
$(document).ready(function() {
    $('.calendar-day.has-events').click(function() {
        var date = $(this).data('date');
        if (!date) {
            alert('Invalid date selected.');
            return;
        }

        $.ajax({
            url: 'fetch_transactions.php',
            method: 'POST',
            data: { date: date },
            success: function(response) {
                $('#transactionModalLabel').text('Transactions for ' + date);
                $('#transactionDetails').html(response);
                $('#transactionModal').modal('show');
            },
            error: function(xhr, status, error) {
                alert('Failed to load transactions: ' + error);
            }
        });
    });
});
</script>
<?php include '../../includes/footer.php'; ?>