<?php
require_once '../../config/db.php';

// Enable error logging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Optional supplier ID filter
$supplier_id = isset($_GET['supplier_id']) ? (int) $_GET['supplier_id'] : 0;

// Query to get all unpaid invoices
$query = "
    SELECT 
        i.id, 
        i.invoice_number, 
        i.invoice_date,
        i.due_date,
        i.total_payable, 
        i.status,
        s.name AS supplier_name, 
        b.branch_name,
        i.supplier_id,
        IFNULL(pa.total_paid, 0) AS paid,
        IFNULL(cn.total_credit, 0) AS credit,
        (i.total_payable - IFNULL(pa.total_paid, 0) - IFNULL(cn.total_credit, 0)) AS balance
    FROM invoices i
    JOIN suppliers s ON i.supplier_id = s.id
    JOIN branches b ON i.branch_id = b.id
    LEFT JOIN (
        SELECT invoice_id, SUM(amount_allocated) AS total_paid
        FROM payment_allocations
        WHERE invoice_id IS NOT NULL
        GROUP BY invoice_id
    ) pa ON pa.invoice_id = i.id
    LEFT JOIN (
        SELECT invoice_id, SUM(amount) AS total_credit
        FROM credit_notes
        WHERE invoice_id IS NOT NULL
        GROUP BY invoice_id
    ) cn ON cn.invoice_id = i.id
    WHERE i.status != 'Paid'
    AND (i.total_payable - IFNULL(pa.total_paid, 0) - IFNULL(cn.total_credit, 0)) > 0
";

if ($supplier_id > 0) {
    $query .= " AND i.supplier_id = ?";
    $stmt = $conn->prepare($query);
    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        http_response_code(500);
        echo json_encode(["error" => "Query preparation failed"]);
        exit;
    }
    $stmt->bind_param("i", $supplier_id);
    $stmt->execute();
    $result = $stmt->get_result();
} else {
    $result = $conn->query($query);
    if (!$result) {
        error_log("Query failed: " . $conn->error);
        http_response_code(500);
        echo json_encode(["error" => "Query execution failed"]);
        exit;
    }
}

// Prepare result array
$invoices = [];
while ($row = $result->fetch_assoc()) {
    $invoices[] = [
        'id' => $row['id'],
        'invoice_number' => $row['invoice_number'],
        'invoice_date' => $row['invoice_date'],
        'due_date' => $row['due_date'],
        'branch_name' => $row['branch_name'],
        'status' => $row['status'],
        'supplier_name' => $row['supplier_name'],
        'supplier_id' => $row['supplier_id'],
        'total_payable' => number_format($row['total_payable'], 2),
        'paid' => number_format($row['paid'], 2),
        'credit' => number_format($row['credit'], 2),
        'balance' => number_format($row['balance'], 2)
    ];
}

header('Content-Type: application/json');
echo json_encode($invoices);
?>