<?php
// Ensure no output before JSON
ob_start();

// Enable error logging
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', '../../logs/error.log');

try {
    // Verify file exists
    if (!file_exists('../../config/db.php')) {
        throw new Exception('Database config file not found at ../../config/db.php');
    }
    require_once '../../config/db.php';

    // Check database connection
    if (!$conn) {
        throw new Exception('Database connection is null');
    }
    if ($conn->connect_error) {
        throw new Exception('Database connection failed: ' . $conn->connect_error);
    }

    // Get parameters
    $payment_date = $_GET['payment_date'] ?? '';
    $supplier_id = isset($_GET['supplier_id']) ? (int)$_GET['supplier_id'] : 0;
    $from_date = $_GET['from_date'] ?? '';
    $to_date = $_GET['to_date'] ?? '';

    // Log request
    error_log("get_payments.php: Request - payment_date: $payment_date, supplier_id: $supplier_id, from_date: $from_date, to_date: $to_date");

    // Validate payment_date
    if (empty($payment_date)) {
        error_log("get_payments.php: Missing payment_date");
        header('Content-Type: application/json');
        echo json_encode([]);
        ob_end_flush();
        exit;
    }

    // Build query conditions
    $conditions = ["DATE(p.payment_date) = ?"];
    $params = [$payment_date];
    $types = 's';

    if ($supplier_id > 0) {
        $conditions[] = "p.supplier_id = ?";
        $params[] = $supplier_id;
        $types .= 'i';
    }
    if (!empty($from_date)) {
        $conditions[] = "DATE(p.payment_date) >= ?";
        $params[] = $from_date;
        $types .= 's';
    }
    if (!empty($to_date)) {
        $conditions[] = "DATE(p.payment_date) <= ?";
        $params[] = $to_date;
        $types .= 's';
    }
    $where = 'WHERE ' . implode(' AND ', $conditions);

    // Log query
    error_log("get_payments.php: Query - $where, params: " . implode(', ', $params));

    // Simplified query to avoid subquery issues
    $query = "
        SELECT p.id, p.supplier_id, s.name AS supplier_name, p.amount, p.payment_date, 
               COALESCE(p.reference, 'N/A') AS reference
        FROM payments p
        JOIN suppliers s ON p.supplier_id = s.id
        $where
        ORDER BY p.payment_date ASC
    ";

    $stmt = $conn->prepare($query);
    if (!$stmt) {
        throw new Exception('Prepare failed: ' . $conn->error);
    }
    $stmt->bind_param($types, ...$params);
    if (!$stmt->execute()) {
        throw new Exception('Execute failed: ' . $stmt->error);
    }
    $result = $stmt->get_result();

    $payments = [];
    while ($row = $result->fetch_assoc()) {
        $payments[] = [
            'id' => $row['id'],
            'supplier_name' => htmlspecialchars($row['supplier_name']),
            'amount' => number_format($row['amount'], 2),
            'payment_date' => $row['payment_date'],
            'reference' => htmlspecialchars($row['reference']),
            'invoice_numbers' => 'None' // Placeholder, as subquery removed
        ];
    }

    error_log("get_payments.php: Found " . count($payments) . " payments for $payment_date");
    header('Content-Type: application/json');
    echo json_encode($payments);

} catch (Exception $e) {
    error_log("get_payments.php: Error - " . $e->getMessage());
    header('Content-Type: application/json', true, 500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
} finally {
    ob_end_flush();
}
exit;