<?php
session_start();
require_once '../../config/db.php';

// ✅ Logging function
function logPaymentAction($conn, $payment_id, $user_id, $action, $remarks = '') {
    $log = $conn->prepare("INSERT INTO payment_logs (payment_id, user_id, action, remarks) VALUES (?, ?, ?, ?)");
    $log->bind_param("iiss", $payment_id, $user_id, $action, $remarks);
    $log->execute();
}

// ✅ CSRF check
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    $_SESSION['error'] = "❌ Invalid CSRF token.";
    header("Location: select_invoices.php");
    exit;
}

// ✅ Validate inputs
if (!isset($_POST['invoice_allocations_json']) || !isset($_POST['allocations'])) {
    $_SESSION['error'] = "No invoice allocations provided.";
    header("Location: select_invoices.php");
    exit;
}

$invoice_allocations = json_decode($_POST['invoice_allocations_json'], true);
$allocations = $_POST['allocations'];

if (!is_array($invoice_allocations) || empty($invoice_allocations) || !is_array($allocations)) {
    $_SESSION['error'] = "Invalid invoice allocations.";
    header("Location: select_invoices.php");
    exit;
}

$supplier_id = (int) $_POST['supplier_id'];
if ($supplier_id <= 0) {
    $_SESSION['error'] = "Invalid supplier ID.";
    header("Location: select_invoices.php");
    exit;
}

foreach ($invoice_allocations as $inv) {
    if ((int) $inv['supplier_id'] !== $supplier_id) {
        $_SESSION['error'] = "All invoices must belong to the same supplier.";
        header("Location: select_invoices.php");
        exit;
    }
}

$payment_amount = (float) ($_POST['payment_amount'] ?? 0);
$payment_method = $_POST['payment_method'] ?? '';
$reference_no = $_POST['reference_no'] ?? '';
$payment_date = $_POST['payment_date'] ?? date('Y-m-d');
$remarks = $_POST['remarks'] ?? '';

if ($payment_amount <= 0 || empty($payment_method) || empty($reference_no) || empty($payment_date)) {
    $_SESSION['error'] = "Invalid payment details.";
    header("Location: add.php");
    exit;
}

// Check for unique reference number
$stmt = $conn->prepare("SELECT id FROM payments WHERE reference_no = ?");
$stmt->bind_param("s", $reference_no);
$stmt->execute();
if ($stmt->get_result()->num_rows > 0) {
    $_SESSION['error'] = "Reference number '$reference_no' already exists.";
    header("Location: add.php");
    exit;
}

// ✅ Validate allocations
$total_allocated = 0;
$validated_allocations = [];
foreach ($allocations as $invoice_id => $amount) {
    $invoice_id = (int) $invoice_id;
    $amount = (float) $amount;

    if ($amount < 0) {
        $_SESSION['error'] = "Allocation amounts cannot be negative.";
        header("Location: add.php");
        exit;
    }

    $stmt = $conn->prepare("
        SELECT 
            i.total_payable,
            IFNULL((SELECT SUM(amount_allocated) FROM payment_allocations WHERE invoice_id = i.id), 0) AS paid,
            IFNULL((SELECT SUM(amount) FROM credit_notes WHERE invoice_id = i.id), 0) AS credit
        FROM invoices i
        WHERE i.id = ?
    ");
    $stmt->bind_param("i", $invoice_id);
    $stmt->execute();
    $invoice = $stmt->get_result()->fetch_assoc();

    if (!$invoice) {
        $_SESSION['error'] = "Invalid invoice ID: $invoice_id.";
        header("Location: add.php");
        exit;
    }

    $balance = $invoice['total_payable'] - $invoice['paid'] - $invoice['credit'];
    if ($amount > $balance) {
        $_SESSION['error'] = "Allocated amount for invoice ID $invoice_id exceeds balance.";
        header("Location: add.php");
        exit;
    }

    $total_allocated += $amount;
    $validated_allocations[$invoice_id] = $amount;
}

if ($total_allocated > $payment_amount) {
    $_SESSION['error'] = "Total allocated amount exceeds payment amount.";
    header("Location: add.php");
    exit;
}

// ✅ Start transaction
$conn->begin_transaction();

try {
    // Insert payment
    $stmt = $conn->prepare("
        INSERT INTO payments (supplier_id, payment_date, reference_no, amount, method, remarks, created_at, updated_at, updated_by)
        VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW(), ?)
    ");
    $user_id = $_SESSION['user_id'];
    $stmt->bind_param("issdssi", $supplier_id, $payment_date, $reference_no, $payment_amount, $payment_method, $remarks, $user_id);
    if (!$stmt->execute()) {
        throw new Exception("Error inserting payment: " . $stmt->error);
    }

    $payment_id = $stmt->insert_id;

    // Log payment creation
    logPaymentAction($conn, $payment_id, $user_id, 'Created', 'Payment recorded and allocated.');

    // Insert allocations
    foreach ($validated_allocations as $invoice_id => $amount) {
        if ($amount > 0) {
            $stmt = $conn->prepare("
                INSERT INTO payment_allocations (payment_id, invoice_id, amount_allocated)
                VALUES (?, ?, ?)
            ");
            $stmt->bind_param("iid", $payment_id, $invoice_id, $amount);
            if (!$stmt->execute()) {
                throw new Exception("Error inserting allocation for invoice $invoice_id: " . $stmt->error);
            }
        }
    }

    // Update invoice status
    foreach ($validated_allocations as $invoice_id => $amount) {
        if ($amount > 0) {
            $stmt = $conn->prepare("
                SELECT 
                    i.total_payable,
                    IFNULL((SELECT SUM(amount_allocated) FROM payment_allocations WHERE invoice_id = i.id), 0) AS paid,
                    IFNULL((SELECT SUM(amount) FROM credit_notes WHERE invoice_id = i.id), 0) AS credit
                FROM invoices i
                WHERE i.id = ?
            ");
            $stmt->bind_param("i", $invoice_id);
            $stmt->execute();
            $invoice = $stmt->get_result()->fetch_assoc();

            $balance = $invoice['total_payable'] - $invoice['paid'] - $invoice['credit'];
            $status = $balance <= 0 ? 'Paid' : 'Draft';

            $stmt = $conn->prepare("UPDATE invoices SET status = ? WHERE id = ?");
            $stmt->bind_param("si", $status, $invoice_id);
            if (!$stmt->execute()) {
                throw new Exception("Error updating invoice status for ID $invoice_id: " . $stmt->error);
            }
        }
    }

    // ✅ COMMIT TRANSACTION
    $conn->commit();

    // ✅ SEND SMS NOTIFICATION with Full Payment Details
    $apiKey    = 'km9vP4VFDEtLoCcQMRKGdqsYrwB0jHpxO5JeA86nN1bW723T';
    $clientId  = 'DELTASTORES';
    $accessKey = 'YOUR_ACCESS_KEY_HERE'; // Replace with actual AccessKey
    $senderId  = 'DELTASTORES';
    $sms_recipient = '254705705888'; // Target number

    // Get supplier name
    $stmt = $conn->prepare("SELECT name FROM suppliers WHERE id = ?");
    $stmt->bind_param("i", $supplier_id);
    $stmt->execute();
    $stmt->bind_result($supplier_name);
    $stmt->fetch();
    $stmt->close();

    $sms_message = "PAYMENT \nSupplier: $supplier_name\nAmount: KES " . number_format($payment_amount, 2) . "\nMethod: $payment_method\nRef: $reference_no\nDate: $payment_date";

    $payload = [
        "SenderId" => $senderId,
        "IsUnicode" => false,
        "IsFlash" => false,
        "ScheduleDateTime" => "",
        "MessageParameters" => [
            [
                "Number" => $sms_recipient,
                "Text" => $sms_message
            ]
        ],
        "ApiKey" => $apiKey,
        "ClientId" => $clientId
    ];

    $ch = curl_init('https://api.onfonmedia.co.ke/v1/sms/SendBulkSMS');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'AccessKey: ' . $accessKey
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    $sms_response = curl_exec($ch);
    $sms_http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    error_log("SMS Sent | HTTP $sms_http_code | Response: $sms_response");

$_SESSION['payment_summary'] = [
  [
    'ref'   => $reference_no,
    'mode'  => ucfirst($payment_method),
    'amount'=> $payment_amount
  ]
];

$_SESSION['paid_invoices'] = [];

foreach ($validated_allocations as $invoice_id => $amount) {
    if ($amount > 0) {
        $_SESSION['paid_invoices'][] = [
            'invoice_id' => $invoice_id,
            'amount'     => $amount
        ];
    }
}

$_SESSION['success'] = "✅ Payment and allocations saved successfully!";
header("Location: payment_success.php");
exit;


} catch (Exception $e) {
    $conn->rollback();
    error_log("Payment Error: " . $e->getMessage());
    $_SESSION['error'] = "Error processing payment. Please try again.";
    header("Location: add.php");
    exit;
}
?>
