<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$supplier_id = isset($_GET['supplier_id']) ? (int) $_GET['supplier_id'] : 0;

// Base query with placeholder for supplier filter
$query = "
    SELECT 
        i.id AS invoice_id,
        i.invoice_number,
        i.invoice_date,
        i.due_date,
        i.total_payable,
        i.status,
        s.name AS supplier_name,
        b.branch_name,
        i.supplier_id,
        IFNULL(pa.total_paid, 0) AS paid,
        IFNULL(cn.total_credit, 0) AS credit,
        (i.total_payable - IFNULL(pa.total_paid, 0) - IFNULL(cn.total_credit, 0)) AS balance
    FROM invoices i
    JOIN suppliers s ON i.supplier_id = s.id
    JOIN branches b ON i.branch_id = b.id
    LEFT JOIN (
        SELECT invoice_id, SUM(amount_allocated) AS total_paid
        FROM payment_allocations
        WHERE invoice_id IS NOT NULL
        GROUP BY invoice_id
    ) pa ON pa.invoice_id = i.id
    LEFT JOIN (
        SELECT invoice_id, SUM(amount) AS total_credit
        FROM credit_notes
        WHERE invoice_id IS NOT NULL
        GROUP BY invoice_id
    ) cn ON cn.invoice_id = i.id
    WHERE i.status != 'Paid'
    AND (i.total_payable - IFNULL(pa.total_paid, 0) - IFNULL(cn.total_credit, 0)) > 0
";

// Add supplier filter if provided
$bind_params = [];
if ($supplier_id > 0) {
    $query .= " AND i.supplier_id = ?";
    $bind_params[] = ['type' => 'i', 'value' => $supplier_id];
}

$query .= " ORDER BY i.due_date ASC";

// Log the query
error_log("SQL Query: " . $query);
if ($supplier_id > 0) {
    error_log("Bind Param: supplier_id = " . $supplier_id);
}

// Execute query
if ($supplier_id > 0) {
    $stmt = $conn->prepare($query);
    if (!$stmt) {
        $error = "Prepare failed: " . $conn->error . " | Query: " . $query;
        error_log($error);
        $_SESSION['error'] = "Database query preparation failed.";
        header("Location: select_invoices.php");
        exit;
    }
    $stmt->bind_param("i", $supplier_id);
    $stmt->execute();
    $invoices = $stmt->get_result();
} else {
    $invoices = $conn->query($query);
    if (!$invoices) {
        $error = "Query failed: " . $conn->error . " | Query: " . $query;
        error_log($error);
        $_SESSION['error'] = "Database query execution failed.";
        header("Location: select_invoices.php");
        exit;
    }
}

// Calculate totals
$total_payable = 0;
$total_paid = 0;
$total_credit = 0;
$total_balance = 0;
$invoices_data = [];
while ($inv = $invoices->fetch_assoc()) {
    $invoices_data[] = $inv;
    $total_payable += $inv['total_payable'];
    $total_paid += $inv['paid'];
    $total_credit += $inv['credit'];
    $total_balance += $inv['balance'];
}
?>

<style>
.table-responsive {
    position: relative;
    max-height: 500px;
    overflow-y: auto;
}
.table tfoot tr {
    position: sticky;
    bottom: 0;
    z-index: 1;
    background-color: #343a40;
    color: white;
}
</style>

<div class="main-content">
    <h4 class="fw-bold text-primary mb-4"><i class="fas fa-file-invoice"></i> Select Invoices to Pay</h4>

    <form method="GET" class="mb-4">
        <div class="row g-2 align-items-center">
            <div class="col-auto"><label class="form-label mb-0">Filter by Supplier:</label></div>
            <div class="col-md-4">
                <select name="supplier_id" class="form-select" onchange="this.form.submit()">
                    <option value="">-- All Suppliers --</option>
                    <?php
                    $suppliers = $conn->query("SELECT id, name FROM suppliers ORDER BY name ASC");
                    while ($s = $suppliers->fetch_assoc()) {
                        echo "<option value='{$s['id']}' " . ($s['id'] == $supplier_id ? 'selected' : '') . ">" . htmlspecialchars($s['name']) . "</option>";
                    }
                    ?>
                </select>
            </div>
        </div>
    </form>

    <form method="POST" action="add.php" onsubmit="return validateInvoiceSelection();">
        <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
        <input type="hidden" name="selected_invoice_json" id="selected_invoice_json">

        <div class="table-responsive mb-3">
            <table class="table table-bordered table-hover align-middle">
                <thead class="table-dark">
                    <tr>
                        <th>Select</th>
                        <th>Invoice No</th>
                        <th>Supplier</th>
                        <th>Branch</th>
                        <th>Status</th>
                        <th>Invoice Date</th>
                        <th>Due Date</th>
                        <th>Total</th>
                        <th>Paid</th>
                        <th>Credit</th>
                        <th>Balance</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (count($invoices_data) > 0): ?>
                        <?php foreach ($invoices_data as $inv): ?>
                            <tr>
                                <td>
                                   <?php
    $isApproved = strtolower($inv['status']) === 'approved';
    $disabled = $isApproved ? '' : 'disabled';
    $tooltip = $isApproved ? '' : 'data-bs-toggle="tooltip" title="Invoice must be approved to pay"';
?>
<input type="checkbox"
    class="form-check-input invoice-checkbox"
    value="<?= $inv['invoice_id'] ?>"
    data-amount="<?= $inv['balance'] ?>"
    data-supplier="<?= $inv['supplier_id'] ?>"
    onchange="updateTotals()"
    <?= $disabled ?> <?= $tooltip ?>>
                                </td>
                                <td><?= htmlspecialchars($inv['invoice_number']) ?></td>
                                <td><?= htmlspecialchars($inv['supplier_name']) ?></td>
                                <td><?= htmlspecialchars($inv['branch_name']) ?></td>
                                <td><span class="badge bg-secondary"><?= $inv['status'] ?></span></td>
                                <td><?= date('d-M-Y', strtotime($inv['invoice_date'])) ?></td>
                                <td><?= $inv['due_date'] ? date('d-M-Y', strtotime($inv['due_date'])) : 'N/A' ?></td>
                                <td class="text-end"> <?= number_format($inv['total_payable'], 2) ?></td>
                                <td class="text-end text-muted"> <?= number_format($inv['paid'], 2) ?></td>
                                <td class="text-end text-warning"> <?= number_format($inv['credit'], 2) ?></td>
                                <td class="text-end fw-bold text-success"> <?= number_format($inv['balance'], 2) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr><td colspan="11" class="text-center text-muted">No unpaid invoices found.</td></tr>
                    <?php endif; ?>
                </tbody>
                <tfoot>
                    <tr class="text-center">
                        <td colspan="7"><strong>Totals</strong></td>
                        <td><strong> <?= number_format($total_payable, 2) ?></strong></td>
                        <td><strong> <?= number_format($total_paid, 2) ?></strong></td>
                        <td><strong> <?= number_format($total_credit, 2) ?></strong></td>
                        <td><strong> <?= number_format($total_balance, 2) ?></strong></td>
                    </tr>
                </tfoot>
            </table>
        </div>

        <div id="allocationSummary" class="mb-4">
            <div class="alert alert-info d-flex justify-content-between">
                <span>✅ Invoices Selected: <strong id="selectedCount">0</strong></span>
                <span>Total Amount: <strong class="text-success" id="selectedTotal"> 0.00</strong></span>
            </div>
        </div>

        <div class="text-end">
            <button type="submit" class="btn btn-primary"><i class="fas fa-arrow-right me-1"></i> Proceed to Payment</button>
        </div>
    </form>
</div>

<script>
function updateTotals() {
    const checkboxes = document.querySelectorAll(".invoice-checkbox:checked");
    let total = 0;
    let selectedSupplier = null;
    const selectedData = [];

    for (const cb of checkboxes) {
        const invoice_id = cb.value;
        const amount = parseFloat(cb.dataset.amount);
        const supplier_id = cb.dataset.supplier;

        if (selectedSupplier && supplier_id !== selectedSupplier) {
            alert("⚠️ Please select invoices from one supplier only.");
            cb.checked = false;
            return;
        }

        selectedSupplier = supplier_id;
        total += amount;
        selectedData.push({ invoice_id, amount, supplier_id });
    }

    document.getElementById("selectedCount").textContent = selectedData.length;
    document.getElementById("selectedTotal").textContent = `KSh ${total.toLocaleString(undefined, { minimumFractionDigits: 2 })}`;
    document.getElementById("selected_invoice_json").value = JSON.stringify(selectedData);
}

function validateInvoiceSelection() {
    const selected = document.querySelectorAll(".invoice-checkbox:checked");
    if (selected.length === 0) {
        alert("⚠️ Please select at least one invoice.");
        return false;
    }
    return true;
}
document.addEventListener('DOMContentLoaded', function () {
  const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
  tooltipTriggerList.map(function (tooltipTriggerEl) {
    return new bootstrap.Tooltip(tooltipTriggerEl);
  });
});
</script>
</body>
</html>