<?php
session_start();
require_once '../../config/db.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error'] = 'Invalid request.';
    header('Location: add.php');
    exit;
}

if (!isset($_SESSION['user_id'], $_SESSION['username'])) {
    $_SESSION['error'] = 'Unauthorized access.';
    header('Location: ../../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$username = $_SESSION['username'];

$branch_id = (int)($_POST['branch_id'] ?? 0);
$type = $_POST['type'] ?? '';
$amount = (float)($_POST['amount'] ?? 0);
$description = trim($_POST['description'] ?? '');
$transaction_date = $_POST['transaction_date'] ?? date('Y-m-d');
$supplier_id = !empty($_POST['supplier_id']) ? (int)$_POST['supplier_id'] : null;
$payee_name = trim($_POST['payee_name'] ?? '');
$collected_by = trim($_POST['collected_by'] ?? '');
$confirm_password = $_POST['print_password'] ?? '';

$form_data = $_POST;

// Validate required fields
if ($branch_id <= 0 || empty($type) || $amount <= 0 || empty($description) || empty($transaction_date) || empty($collected_by)) {
    $_SESSION['error'] = 'Fill all required fields correctly.';
    $_SESSION['form_data'] = $form_data;
    header('Location: add.php');
    exit;
}

if (!$supplier_id && empty($payee_name)) {
    $_SESSION['error'] = 'Enter Payee Name if no Supplier selected.';
    $_SESSION['form_data'] = $form_data;
    header('Location: add.php');
    exit;
}

// Validate password
$user_stmt = $conn->prepare("SELECT password FROM users WHERE id = ?");
$user_stmt->bind_param('i', $user_id);
$user_stmt->execute();
$user_result = $user_stmt->get_result();
$user_row = $user_result->fetch_assoc();

if (!$user_row || !password_verify($confirm_password, $user_row['password'])) {
    $_SESSION['error'] = 'Invalid password confirmation.';
    $_SESSION['form_data'] = $form_data;
    header('Location: add.php');
    exit;
}

// Generate reference number
$today = date('Ymd');
$prefix = 'PC-' . $today . '-';

$last_query = "SELECT reference_no FROM petty_cash_transactions WHERE reference_no LIKE ? ORDER BY reference_no DESC LIMIT 1";
$stmt = $conn->prepare($last_query);
$like_param = $prefix . '%';
$stmt->bind_param('s', $like_param);
$stmt->execute();
$result = $stmt->get_result();
$last_ref = $result->fetch_assoc();

$new_number = $last_ref ? ((int)substr($last_ref['reference_no'], -3)) + 1 : 1;
$reference_no = $prefix . str_pad($new_number, 3, '0', STR_PAD_LEFT);

// Insert transaction
try {
    $insert = $conn->prepare("
        INSERT INTO petty_cash_transactions 
        (branch_id, type, amount, description, reference_no, transaction_date, created_by, status, supplier_id, payee_name, collected_by)
        VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', ?, ?, ?)
    ");
    $insert->bind_param('isisssiiss',
        $branch_id, $type, $amount, $description, $reference_no, $transaction_date,
        $user_id, $supplier_id, $payee_name, $collected_by
    );
    $insert->execute();

    $transaction_id = $conn->insert_id;

    // ✅ Clear session data so form is not refilled on back
    unset($_SESSION['form_data']);
    unset($_SESSION['error']);

    // ✅ Redirect to print page (with print-safe cache headers in that file)
    header('Location: print_petty_cash.php?id=' . $transaction_id);
    exit;
} catch (Exception $e) {
    $_SESSION['error'] = 'Transaction failed: ' . $e->getMessage();
    $_SESSION['form_data'] = $form_data;
    header('Location: add.php');
    exit;
}
