<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// Weekly selector
$today = date('Y-m-d');
$weeks = [];
$start = strtotime('last monday', strtotime(date('Y-01-01')));
for ($i = 0; $i < 52; $i++) {
    $from = date('Y-m-d', strtotime("+$i week", $start));
    $to = date('Y-m-d', strtotime("+6 days", strtotime($from)));
    $weeks[] = ['from' => $from, 'to' => $to, 'label' => "Week " . ($i + 1) . ": $from to $to"];
}

$week_from = $_GET['from'] ?? date('Y-m-d', strtotime('monday this week'));
$week_to = $_GET['to'] ?? date('Y-m-d', strtotime('sunday this week'));

// Accurate allocation-based SQL
$sql = "
SELECT 
  i.id, 
  i.invoice_number, 
  i.invoice_date, 
  i.total_payable, 
  s.name AS supplier_name, 
  b.branch_name,
  COALESCE(r.verified_status, '') AS verified_status,
  COALESCE(r.remarks, '') AS remarks,
  COALESCE(paid.total_paid, 0) AS amount_paid
FROM invoices i
JOIN suppliers s ON i.supplier_id = s.id
JOIN branches b ON i.branch_id = b.id
LEFT JOIN invoice_reconciliation r ON r.invoice_id = i.id
LEFT JOIN (
    SELECT invoice_id, SUM(amount) AS total_paid
    FROM payment_allocations
    GROUP BY invoice_id
) paid ON paid.invoice_id = i.id
WHERE DATE(i.invoice_date) BETWEEN ? AND ?
GROUP BY i.id, i.invoice_number, i.invoice_date, i.total_payable, s.name, b.branch_name, r.verified_status, r.remarks, paid.total_paid
ORDER BY i.invoice_date ASC
";

$stmt = $conn->prepare($sql);
if (!$stmt) {
    die("<div class='alert alert-danger'>SQL Error: " . $conn->error . "</div>");
}
$stmt->bind_param('ss', $week_from, $week_to);
$stmt->execute();
$result = $stmt->get_result();
?>

<div class="main-content">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h4 class="fw-bold text-primary"><i class="fas fa-check-circle me-2"></i> Invoice Reconciliation (Physical + Payment)</h4>
  </div>

  <form method="GET" class="row g-3 mb-4">
    <div class="col-md-4">
      <label class="form-label">Select Week</label>
      <select class="form-select" onchange="location.href=this.value">
        <?php foreach ($weeks as $w): 
          $url = "?from={$w['from']}&to={$w['to']}";
          $selected = ($w['from'] == $week_from) ? 'selected' : '';
        ?>
          <option value="<?= $url ?>" <?= $selected ?>><?= $w['label'] ?></option>
        <?php endforeach; ?>
      </select>
    </div>
  </form>

  <?php if (isset($_SESSION['success'])): ?>
    <div class="alert alert-success"><?= $_SESSION['success']; unset($_SESSION['success']); ?></div>
  <?php endif; ?>

  <form action="save_reconciliation.php" method="POST">
    <input type="hidden" name="week_from" value="<?= $week_from ?>">
    <input type="hidden" name="week_to" value="<?= $week_to ?>">

    <div class="table-responsive">
      <table class="table table-bordered align-middle small table-hover">
        <thead class="table-dark text-center">
          <tr>
            <th>#</th>
            <th>Invoice No</th>
            <th>Supplier</th>
            <th>Branch</th>
            <th>Date</th>
            <th>Total</th>
            <th>Paid</th>
            <th>Balance</th>
            <th>Match</th>
            <th>Reconciliation</th>
            <th>Remarks</th>
          </tr>
        </thead>
        <tbody>
          <?php if ($result->num_rows): $i = 1; while($row = $result->fetch_assoc()): 
            $balance = $row['total_payable'] - $row['amount_paid'];
            $match_status = $row['amount_paid'] == 0 ? '❌ Unpaid' : ($balance > 0 ? '⚠️ Partial' : '✅ Paid');
          ?>
          <tr>
            <td><?= $i++ ?></td>
            <td>
              <?= htmlspecialchars($row['invoice_number']) ?>
              <input type="hidden" name="invoice_ids[]" value="<?= $row['id'] ?>">
            </td>
            <td><?= htmlspecialchars($row['supplier_name']) ?></td>
            <td><?= htmlspecialchars($row['branch_name']) ?></td>
            <td><?= date('d-M-Y', strtotime($row['invoice_date'])) ?></td>
            <td class="text-end">KSh <?= number_format($row['total_payable'], 2) ?></td>
            <td class="text-end text-success">KSh <?= number_format($row['amount_paid'], 2) ?></td>
            <td class="text-end <?= $balance != 0 ? 'text-danger' : 'text-muted' ?>">KSh <?= number_format($balance, 2) ?></td>
            <td class="text-center"><?= $match_status ?></td>
            <td>
              <select name="status_<?= $row['id'] ?>" class="form-select form-select-sm" required>
                <option value="">-- Select --</option>
                <option value="Verified" <?= $row['verified_status'] == 'Verified' ? 'selected' : '' ?>>✅ Verified</option>
                <option value="Not Found" <?= $row['verified_status'] == 'Not Found' ? 'selected' : '' ?>>❌ Not Found</option>
                <option value="Mismatch" <?= $row['verified_status'] == 'Mismatch' ? 'selected' : '' ?>>⚠️ Mismatch</option>
              </select>
            </td>
            <td><input type="text" name="remarks_<?= $row['id'] ?>" class="form-control form-control-sm" value="<?= htmlspecialchars($row['remarks']) ?>"></td>
          </tr>
          <?php endwhile; else: ?>
            <tr><td colspan="11" class="text-center text-muted">No invoices found for this week.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>

    <div class="text-end mt-3">
      <button type="submit" class="btn btn-success"><i class="fas fa-save me-1"></i> Save Reconciliation</button>
    </div>
  </form>
</div>

<?php include '../../includes/footer.php'; ?>
