<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// Filters
$supplier_id = $_GET['supplier_id'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Build WHERE clause
$where = "1";
$params = [];
if ($supplier_id !== '') {
    $where .= " AND i.supplier_id = ?";
    $params[] = $supplier_id;
}
if ($date_from !== '') {
    $where .= " AND i.invoice_date >= ?";
    $params[] = $date_from;
}
if ($date_to !== '') {
    $where .= " AND i.invoice_date <= ?";
    $params[] = $date_to;
}

// Build query
$sql = "
    SELECT 
        s.id AS supplier_id, s.name AS supplier_name,
        SUM(i.total_payable) AS invoice_total,
        IFNULL((
            SELECT SUM(pa.amount_allocated)
            FROM payment_allocations pa
            JOIN payments p ON pa.payment_id = p.id
            WHERE pa.invoice_id = i.id
        ), 0) AS paid_total,
        IFNULL((
            SELECT SUM(cn.amount)
            FROM credit_notes cn
            WHERE cn.invoice_id = i.id
        ), 0) AS credit_total,
        MAX(i.invoice_date) AS last_invoice_date
    FROM invoices i
    JOIN suppliers s ON i.supplier_id = s.id
    WHERE $where
    GROUP BY s.id
    ORDER BY s.name
";

$stmt = $conn->prepare($sql);
if (!empty($params)) {
    $types = str_repeat("s", count($params));
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

$suppliers = [];
while ($row = $result->fetch_assoc()) {
    $paid = $row['paid_total'];
    $credit = $row['credit_total'];
    $balance = $row['invoice_total'] - $paid - $credit;
    $row['balance'] = $balance;

    $days = (new DateTime())->diff(new DateTime($row['last_invoice_date']))->days;
    if ($balance <= 0) {
        $row['bucket'] = 'Cleared';
    } elseif ($days <= 30) {
        $row['bucket'] = '0-30';
    } elseif ($days <= 60) {
        $row['bucket'] = '31-60';
    } elseif ($days <= 90) {
        $row['bucket'] = '61-90';
    } else {
        $row['bucket'] = '90+';
    }

    $suppliers[] = $row;
}
?>

<div class="container mt-4">
  <h4 class="text-primary mb-3"><i class="fas fa-balance-scale"></i> Supplier Ledger Reconciliation</h4>

  <form method="get" class="row g-3 mb-4">
    <div class="col-md-4">
      <label>Supplier</label>
      <select name="supplier_id" class="form-select">
        <option value="">-- All Suppliers --</option>
        <?php
        $suppliers_list = $conn->query("SELECT id, name FROM suppliers ORDER BY name");
        while ($s = $suppliers_list->fetch_assoc()):
        ?>
          <option value="<?= $s['id'] ?>" <?= $s['id'] == $supplier_id ? 'selected' : '' ?>><?= htmlspecialchars($s['name']) ?></option>
        <?php endwhile; ?>
      </select>
    </div>
    <div class="col-md-3">
      <label>From</label>
      <input type="date" name="date_from" value="<?= $date_from ?>" class="form-control">
    </div>
    <div class="col-md-3">
      <label>To</label>
      <input type="date" name="date_to" value="<?= $date_to ?>" class="form-control">
    </div>
    <div class="col-md-2 d-grid">
      <label>&nbsp;</label>
      <button class="btn btn-primary">Filter</button>
    </div>
  </form>

  <div class="table-responsive">
    <table class="table table-bordered table-hover small">
      <thead class="table-dark">
        <tr>
          <th>Supplier</th>
          <th>Invoiced</th>
          <th>Paid</th>
          <th>Credit</th>
          <th>Balance</th>
          <th>Aging Bucket</th>
        </tr>
      </thead>
      <tbody>
        <?php if (count($suppliers) > 0): ?>
          <?php foreach ($suppliers as $s): ?>
            <tr>
              <td><?= htmlspecialchars($s['supplier_name']) ?></td>
              <td class="text-end text-muted">KSh <?= number_format($s['invoice_total'], 2) ?></td>
              <td class="text-end text-success">KSh <?= number_format($s['paid_total'], 2) ?></td>
              <td class="text-end text-warning">KSh <?= number_format($s['credit_total'], 2) ?></td>
              <td class="text-end fw-bold <?= $s['balance'] > 0 ? 'text-danger' : 'text-success' ?>">KSh <?= number_format($s['balance'], 2) ?></td>
              <td class="text-center"><?= $s['bucket'] ?></td>
            </tr>
          <?php endforeach; ?>
        <?php else: ?>
          <tr><td colspan="6" class="text-center text-muted">No results found.</td></tr>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>
