<?php
// analytics_sales_purchases_payments.php
// Optimized charts + Branch-wise analytics
// ---------------------------------------------------------------

session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

// ===== Helpers =====
function fmt($n) { return number_format((float)$n, 2); }
function bindAndExec(mysqli $conn, string $sql, string $types = '', array $params = []) {
    $stmt = $conn->prepare($sql);
    if (!$stmt) { die("SQL error: " . $conn->error); }
    if ($types !== '' && !empty($params)) { $stmt->bind_param($types, ...$params); }
    $stmt->execute();
    return $stmt->get_result();
}

// ===== Filters =====
$today = date('Y-m-d');
$from = $_GET['from'] ?? date('Y-m-01');
$to   = $_GET['to']   ?? date('Y-m-d');
$branch_id = isset($_GET['branch_id']) ? (int)$_GET['branch_id'] : 0;
$supplier_id = isset($_GET['supplier_id']) ? (int)$_GET['supplier_id'] : 0;

// (Optional) Enable branch-level Purchases/Payments breakdowns if those tables carry branch_id columns
// Set to true only if your invoices/payments tables have branch_id and are populated correctly.
$HAS_BRANCH_IN_PURCHASES = false;
$HAS_BRANCH_IN_PAYMENTS  = false;

$branches_rs = $conn->query("SELECT id, branch_name FROM branches ORDER BY branch_name");
$suppliers_rs = $conn->query("SELECT id, name FROM suppliers ORDER BY name");

// Build date params
$baseTypes = 'ss';
$baseParams = [$from, $to];

// --- SALES (Till Reports) ---
$salesTypes = $baseTypes;
$salesParams = $baseParams;
$salesBranchWhere = '';
if ($branch_id > 0) {
    $salesBranchWhere = " AND h.branch_id = ? ";
    $salesTypes .= 'i';
    $salesParams[] = $branch_id;
}

$salesSQL = "
    WITH cheque_sum AS (
        SELECT report_id, SUM(amount) AS cheques_total
        FROM till_report_cheques
        GROUP BY report_id
    )
    SELECT 
        DATE(h.report_date) AS dte,
        b.id AS branch_id,
        b.branch_name,
        SUM(
          (d.note_1000 + d.note_500 + d.note_200 + d.note_100 + d.note_50)
          + d.coins
          + (d.drop1 + d.drop2 + d.drop3)
        ) AS cash_total,
        SUM(d.mpesa + d.stk) AS mpesa_total,
        SUM(d.visa) AS visa_total,
        SUM(d.credit) AS credit_total,
        SUM(d.declared_total) AS declared_total,
        SUM(COALESCE(c.cheques_total,0)) AS cheques_total
    FROM till_report_headers h
    JOIN branches b ON b.id = h.branch_id
    JOIN till_report_details d ON d.report_id = h.id
    LEFT JOIN cheque_sum c ON c.report_id = h.id
    WHERE h.report_date BETWEEN ? AND ?
    $salesBranchWhere
    GROUP BY DATE(h.report_date), b.id, b.branch_name
    ORDER BY dte ASC, b.branch_name ASC
";
$salesRes = bindAndExec($conn, $salesSQL, $salesTypes, $salesParams);

// --- PURCHASES (Supplier Invoices) ---
$purTypes = $baseTypes;
$purParams = $baseParams;
$purWhereExtra = '';
if ($supplier_id > 0) {
    $purWhereExtra .= " AND i.supplier_id = ? ";
    $purTypes .= 'i';
    $purParams[] = $supplier_id;
}
if ($HAS_BRANCH_IN_PURCHASES && $branch_id > 0) {
    $purWhereExtra .= " AND i.branch_id = ? ";
    $purTypes .= 'i';
    $purParams[] = $branch_id;
}
$purchaseSQL = $HAS_BRANCH_IN_PURCHASES
? "
    SELECT DATE(i.invoice_date) AS dte, i.branch_id, b.branch_name, SUM(i.total_amount) AS purchases_total
    FROM invoices i
    JOIN branches b ON b.id = i.branch_id
    WHERE i.invoice_date BETWEEN ? AND ?
    $purWhereExtra
    GROUP BY DATE(i.invoice_date), i.branch_id, b.branch_name
    ORDER BY dte ASC, b.branch_name ASC
"
: "
    SELECT DATE(i.invoice_date) AS dte, SUM(i.total_amount) AS purchases_total
    FROM invoices i
    WHERE i.invoice_date BETWEEN ? AND ?
    $purWhereExtra
    GROUP BY DATE(i.invoice_date)
    ORDER BY dte ASC
";
$purchaseRes = bindAndExec($conn, $purchaseSQL, $purTypes, $purParams);

// --- PAYMENTS (to Suppliers) ---
$payTypes = $baseTypes;
$payParams = $baseParams;
$payWhereExtra = '';
if ($supplier_id > 0) { $payWhereExtra .= " AND p.supplier_id = ? "; $payTypes .= 'i'; $payParams[] = $supplier_id; }
if ($HAS_BRANCH_IN_PAYMENTS && $branch_id > 0) {
    $payWhereExtra .= " AND p.branch_id = ? ";
    $payTypes .= 'i';
    $payParams[] = $branch_id;
}
$paymentSQL = $HAS_BRANCH_IN_PAYMENTS
? "
    SELECT DATE(p.payment_date) AS dte, p.branch_id, b.branch_name, SUM(p.amount) AS payments_total
    FROM payments p
    JOIN branches b ON b.id = p.branch_id
    WHERE p.payment_date BETWEEN ? AND ?
    $payWhereExtra
    GROUP BY DATE(p.payment_date), p.branch_id, b.branch_name
    ORDER BY dte ASC, b.branch_name ASC
"
: "
    SELECT DATE(p.payment_date) AS dte, SUM(p.amount) AS payments_total
    FROM payments p
    WHERE p.payment_date BETWEEN ? AND ?
    $payWhereExtra
    GROUP BY DATE(p.payment_date)
    ORDER BY dte ASC
";
$paymentRes = bindAndExec($conn, $paymentSQL, $payTypes, $payParams);

// --- PETTY CASH ---
$pcTypes = $baseTypes;
$pcParams = $baseParams;
$pcWhere = '';
if ($branch_id > 0) { $pcWhere .= " AND pct.branch_id = ? "; $pcTypes .= 'i'; $pcParams[] = $branch_id; }

$pettySQL = "
    SELECT DATE(pct.transaction_date) AS dte, pct.branch_id, b.branch_name, SUM(pct.amount) AS petty_total
    FROM petty_cash_transactions pct
    LEFT JOIN branches b ON b.id = pct.branch_id
    WHERE pct.transaction_date BETWEEN ? AND ?
    $pcWhere
    GROUP BY DATE(pct.transaction_date), pct.branch_id, b.branch_name
    ORDER BY dte ASC, b.branch_name ASC
";
$pettyRes = bindAndExec($conn, $pettySQL, $pcTypes, $pcParams);

// ===== Merge global timeseries (ALL branches) =====
$dateset = [];
$totals = [
    'sales_actual' => 0, 'sales_declared' => 0,
    'purchases' => 0, 'payments' => 0, 'petty' => 0
];

// Also build Branch-wise sets for charts
$branches_set = [];              // branch_id => branch_name
$branch_dateset = [];            // branch_id => date => metrics
$branch_totals = [];             // branch_id => totals array

$push_branch = function($bid, $bname) use (&$branches_set, &$branch_dateset, &$branch_totals) {
    if (!$bid) return;
    if (!isset($branches_set[$bid])) $branches_set[$bid] = $bname ?: 'Unknown';
    if (!isset($branch_dateset[$bid])) $branch_dateset[$bid] = [];
    if (!isset($branch_totals[$bid])) {
        $branch_totals[$bid] = [
            'sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0
        ];
    }
};

// Sales (build both global and per-branch)
while ($r = $salesRes->fetch_assoc()) {
    $d = $r['dte'];
    $bid = (int)$r['branch_id'];
    $bname = $r['branch_name'];
    $actual = (float)$r['cash_total'] + (float)$r['mpesa_total'] + (float)$r['visa_total'] + (float)$r['cheques_total'] + (float)$r['credit_total'];
    $declared = (float)$r['declared_total'];

    if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
    $dateset[$d]['sales_actual'] += $actual;
    $dateset[$d]['sales_declared'] += $declared;

    $totals['sales_actual'] += $actual;
    $totals['sales_declared'] += $declared;

    // branch
    $push_branch($bid, $bname);
    if (!isset($branch_dateset[$bid][$d])) $branch_dateset[$bid][$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
    $branch_dateset[$bid][$d]['sales_actual'] += $actual;
    $branch_dateset[$bid][$d]['sales_declared'] += $declared;
    $branch_totals[$bid]['sales_actual'] += $actual;
    $branch_totals[$bid]['sales_declared'] += $declared;
}

// Purchases
if ($HAS_BRANCH_IN_PURCHASES) {
    while ($r = $purchaseRes->fetch_assoc()) {
        $d = $r['dte'];
        $bid = (int)$r['branch_id'];
        $bname = $r['branch_name'];
        $val = (float)$r['purchases_total'];

        if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
        $dateset[$d]['purchases'] += $val;
        $totals['purchases'] += $val;

        $push_branch($bid, $bname);
        if (!isset($branch_dateset[$bid][$d])) $branch_dateset[$bid][$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
        $branch_dateset[$bid][$d]['purchases'] += $val;
        $branch_totals[$bid]['purchases'] += $val;
    }
} else {
    while ($r = $purchaseRes->fetch_assoc()) {
        $d = $r['dte'];
        $val = (float)$r['purchases_total'];
        if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
        $dateset[$d]['purchases'] += $val;
        $totals['purchases'] += $val;
    }
}

// Payments
if ($HAS_BRANCH_IN_PAYMENTS) {
    while ($r = $paymentRes->fetch_assoc()) {
        $d = $r['dte'];
        $bid = (int)$r['branch_id'];
        $bname = $r['branch_name'];
        $val = (float)$r['payments_total'];

        if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
        $dateset[$d]['payments'] += $val;
        $totals['payments'] += $val;

        $push_branch($bid, $bname);
        if (!isset($branch_dateset[$bid][$d])) $branch_dateset[$bid][$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
        $branch_dateset[$bid][$d]['payments'] += $val;
        $branch_totals[$bid]['payments'] += $val;
    }
} else {
    while ($r = $paymentRes->fetch_assoc()) {
        $d = $r['dte'];
        $val = (float)$r['payments_total'];
        if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
        $dateset[$d]['payments'] += $val;
        $totals['payments'] += $val;
    }
}

// Petty cash (always has branch)
while ($r = $pettyRes->fetch_assoc()) {
    $d = $r['dte'];
    $bid = (int)$r['branch_id'];
    $bname = $r['branch_name'];
    $val = (float)$r['petty_total'];

    if (!isset($dateset[$d])) $dateset[$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
    $dateset[$d]['petty'] += $val;
    $totals['petty'] += $val;

    $push_branch($bid, $bname);
    if (!isset($branch_dateset[$bid][$d])) $branch_dateset[$bid][$d] = ['sales_actual'=>0,'sales_declared'=>0,'purchases'=>0,'payments'=>0,'petty'=>0];
    $branch_dateset[$bid][$d]['petty'] += $val;
    $branch_totals[$bid]['petty'] += $val;
}

// Sort by date globally and per-branch
ksort($dateset);
foreach ($branch_dateset as $bid => $rows) { ksort($branch_dateset[$bid]); }

// Global KPIs
$gross_margin_proxy = $totals['sales_declared'] - $totals['purchases']; // proxy
$cash_gap = $totals['sales_actual'] - ($totals['payments'] + $totals['petty']); // inflow vs outflow

// Data for Chart.js (global)
$labels = array_keys($dateset);
$salesActualData = array_map(fn($d)=> round($d['sales_actual'],2), $dateset);
$salesDeclaredData = array_map(fn($d)=> round($d['sales_declared'],2), $dateset);
$purchaseData = array_map(fn($d)=> round($d['purchases'],2), $dateset);
$paymentData = array_map(fn($d)=> round($d['payments'],2), $dateset);
$pettyData = array_map(fn($d)=> round($d['petty'],2), $dateset);

// Branch-wise payloads for JS
$branchPayload = [];
foreach ($branch_dateset as $bid => $rows) {
    $branchPayload[$bid] = [
        'name' => $branches_set[$bid] ?? 'Unknown',
        'labels' => array_keys($rows),
        'sales_actual' => array_map(fn($r)=> round($r['sales_actual'],2), $rows),
        'sales_declared' => array_map(fn($r)=> round($r['sales_declared'],2), $rows),
        'purchases' => array_map(fn($r)=> round($r['purchases'],2), $rows),
        'payments' => array_map(fn($r)=> round($r['payments'],2), $rows),
        'petty' => array_map(fn($r)=> round($r['petty'],2), $rows),
        'totals' => $branch_totals[$bid],
    ];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Analytics — Sales vs Purchases vs Payments (Branch-wise)</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- BS + Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
  <!-- Chart.js -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
  <style>
    body { background:#f5f7fb; }
    .kpi-card { border:0; border-radius: 14px; box-shadow: 0 6px 18px rgba(0,0,0,.06); }
    .kpi-card .card-body { padding: 14px 16px; }
    .kpi-value { font-weight: 800; font-size: 1.15rem; }
    .kpi-label { font-size: .85rem; color:#6c757d; }
    .chip { font-size:.72rem; padding:.12rem .45rem; border-radius: 10px; }
    .table-sticky thead th { position: sticky; top: 0; background: #fff; z-index: 1; }
    .mini-help { font-size: 12px; color:#6c757d; }
    /* === Space-optimized chart containers === */
    .chart-wrap {
      height: 60vh;         /* uses 60% of viewport height */
      min-height: 420px;    /* keeps it readable on short screens */
      position: relative;
    }
    .chart-wrap-sm {
      height: 42vh;
      min-height: 300px;
      position: relative;
    }
    /* Responsive grid for branch charts */
    .branch-grid {
      display: grid;
      grid-template-columns: repeat(auto-fill, minmax(380px, 1fr));
      gap: 14px;
    }
    @media (max-width: 576px){
      .chart-wrap { height: 52vh; min-height: 340px; }
      .chart-wrap-sm { height: 38vh; min-height: 260px; }
    }
    @media print { .no-print { display:none; } body{ background:#fff; } }
  </style>
</head>
<body>
<div class="container-fluid py-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0 fw-bold text-primary">
      <i class="fa-solid fa-chart-line me-2"></i>Sales vs Purchases vs Payments — Analytics
    </h5>
    <div class="no-print">
      <button onclick="window.print()" class="btn btn-outline-dark btn-sm">
        <i class="fas fa-print me-1"></i> Print
      </button>
    </div>
  </div>

  <!-- Filters -->
  <div class="card shadow-sm mb-3">
    <div class="card-body">
      <form method="GET" class="row g-2 align-items-end">
        <div class="col-md-2">
          <label class="form-label">From</label>
          <input type="date" name="from" value="<?= htmlspecialchars($from) ?>" class="form-control form-control-sm">
        </div>
        <div class="col-md-2">
          <label class="form-label">To</label>
          <input type="date" name="to" value="<?= htmlspecialchars($to) ?>" class="form-control form-control-sm">
        </div>
        <div class="col-md-3">
          <label class="form-label">Branch</label>
          <select name="branch_id" class="form-select form-select-sm">
            <option value="">-- All Branches --</option>
            <?php while ($b = $branches_rs->fetch_assoc()): ?>
              <option value="<?= $b['id'] ?>" <?= $branch_id == $b['id'] ? 'selected' : '' ?>>
                <?= htmlspecialchars($b['branch_name']) ?>
              </option>
            <?php endwhile; ?>
          </select>
          <div class="mini-help">Affects Sales & Petty Cash. Toggle flags to enable Purchases/Payments per-branch.</div>
        </div>
        <div class="col-md-3">
          <label class="form-label">Supplier</label>
          <select name="supplier_id" class="form-select form-select-sm">
            <option value="">-- All Suppliers --</option>
            <?php while ($s = $suppliers_rs->fetch_assoc()): ?>
              <option value="<?= $s['id'] ?>" <?= $supplier_id == $s['id'] ? 'selected' : '' ?>>
                <?= htmlspecialchars($s['name']) ?>
              </option>
            <?php endwhile; ?>
          </select>
          <div class="mini-help">Filters Purchases & Payments only.</div>
        </div>
        <div class="col-md-2">
          <button class="btn btn-primary btn-sm w-100"><i class="fas fa-filter me-1"></i> Apply</button>
        </div>
      </form>
    </div>
  </div>

  <!-- KPIs -->
  <div class="row g-3 mb-3">
    <div class="col-md-2">
      <div class="card kpi-card"><div class="card-body">
        <div class="kpi-label">Sales (Actual)</div>
        <div class="kpi-value text-success">KSh <?= fmt($totals['sales_actual']) ?></div>
        <span class="chip bg-light text-success"><i class="fa-solid fa-circle me-1"></i> Cash+MPESA+VISA+Chq+Credit</span>
      </div></div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card"><div class="card-body">
        <div class="kpi-label">Sales (POS Expected)</div>
        <div class="kpi-value text-primary">KSh <?= fmt($totals['sales_declared']) ?></div>
        <span class="chip bg-light text-primary"><i class="fa-regular fa-rectangle-list me-1"></i> Declared</span>
      </div></div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card"><div class="card-body">
        <div class="kpi-label">Purchases</div>
        <div class="kpi-value text-danger">KSh <?= fmt($totals['purchases']) ?></div>
        <span class="chip bg-light text-danger"><i class="fa-solid fa-file-invoice me-1"></i> Invoices</span>
      </div></div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card"><div class="card-body">
        <div class="kpi-label">Payments</div>
        <div class="kpi-value text-warning">KSh <?= fmt($totals['payments']) ?></div>
        <span class="chip bg-light text-warning"><i class="fa-solid fa-money-bill-transfer me-1"></i> Suppliers</span>
      </div></div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card"><div class="card-body">
        <div class="kpi-label">Petty Cash</div>
        <div class="kpi-value text-secondary">KSh <?= fmt($totals['petty']) ?></div>
        <span class="chip bg-light text-secondary"><i class="fa-solid fa-sack-dollar me-1"></i> Expenses</span>
      </div></div>
    </div>
    <div class="col-md-2">
      <div class="card kpi-card"><div class="card-body">
        <div class="kpi-label">Gross Margin (Proxy)</div>
        <div class="kpi-value <?= $gross_margin_proxy >=0 ? 'text-success' : 'text-danger' ?>">KSh <?= fmt($gross_margin_proxy) ?></div>
        <span class="chip bg-light"><i class="fa-solid fa-scale-balanced me-1"></i> Declared − Purchases</span>
      </div></div>
    </div>
  </div>

  <!-- Global Trend (space-optimized) -->
  <div class="card shadow-sm mb-4">
    <div class="card-body">
      <div class="d-flex justify-content-between align-items-center mb-2">
        <h6 class="mb-0 fw-bold"><i class="fa-solid fa-chart-column me-2"></i>Daily Trend — All Branches</h6>
        <span class="mini-help">Bars: Sales Actual/Declared | Lines: Purchases, Payments, Petty</span>
      </div>
      <div class="chart-wrap">
        <canvas id="comboChart"></canvas>
      </div>
    </div>
  </div>

  <!-- Branch-wise Tabs -->
  <ul class="nav nav-pills mb-3 no-print" id="branchTab" role="tablist">
    <li class="nav-item" role="presentation">
      <button class="nav-link active" id="all-branches-tab" data-bs-toggle="pill" data-bs-target="#all-branches" type="button" role="tab" aria-controls="all-branches" aria-selected="true">
        <i class="fa-solid fa-layer-group me-1"></i> All Branches Grid
      </button>
    </li>
    <?php foreach ($branches_set as $bid => $bname): ?>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="branch-<?= $bid ?>-tab" data-bs-toggle="pill" data-bs-target="#branch-<?= $bid ?>" type="button" role="tab" aria-controls="branch-<?= $bid ?>" aria-selected="false">
          <i class="fa-solid fa-store me-1"></i><?= htmlspecialchars($bname) ?>
        </button>
      </li>
    <?php endforeach; ?>
  </ul>

  <div class="tab-content" id="branchTabContent">
    <!-- Grid with small multiples -->
    <div class="tab-pane fade show active" id="all-branches" role="tabpanel" aria-labelledby="all-branches-tab">
      <?php if (empty($branches_set)): ?>
        <div class="alert alert-info">No branch-level data found for this range.</div>
      <?php else: ?>
        <div class="branch-grid">
          <?php foreach ($branches_set as $bid => $bname): ?>
            <div class="card shadow-sm">
              <div class="card-body">
                <div class="d-flex justify-content-between align-items-center mb-2">
                  <h6 class="mb-0 fw-bold"><i class="fa-solid fa-store me-1"></i><?= htmlspecialchars($bname) ?></h6>
                  <?php $bt = $branch_totals[$bid]; ?>
                  <span class="mini-help">
                    Sales: <strong class="text-success">KSh <?= fmt($bt['sales_actual']) ?></strong>
                    · Petty: <strong class="text-secondary">KSh <?= fmt($bt['petty']) ?></strong>
                  </span>
                </div>
                <div class="chart-wrap-sm">
                  <canvas id="branchChart<?= $bid ?>"></canvas>
                </div>
              </div>
            </div>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>
    </div>

    <!-- One big chart per branch (uses full height) -->
    <?php foreach ($branches_set as $bid => $bname): ?>
      <div class="tab-pane fade" id="branch-<?= $bid ?>" role="tabpanel" aria-labelledby="branch-<?= $bid ?>-tab">
        <div class="card shadow-sm">
          <div class="card-body">
            <div class="d-flex justify-content-between align-items-center mb-2">
              <h6 class="mb-0 fw-bold"><i class="fa-solid fa-store me-1"></i><?= htmlspecialchars($bname) ?> — Daily Trend</h6>
              <?php $bt = $branch_totals[$bid]; ?>
              <span class="mini-help">
                Sales Actual: <strong class="text-success">KSh <?= fmt($bt['sales_actual']) ?></strong>
                · Declared: <strong class="text-primary">KSh <?= fmt($bt['sales_declared']) ?></strong>
                <?php if ($HAS_BRANCH_IN_PURCHASES): ?> · Purchases: <strong class="text-danger">KSh <?= fmt($bt['purchases']) ?></strong><?php endif; ?>
                <?php if ($HAS_BRANCH_IN_PAYMENTS): ?> · Payments: <strong class="text-warning">KSh <?= fmt($bt['payments']) ?></strong><?php endif; ?>
                · Petty: <strong class="text-secondary">KSh <?= fmt($bt['petty']) ?></strong>
              </span>
            </div>
            <div class="chart-wrap">
              <canvas id="branchBigChart<?= $bid ?>"></canvas>
            </div>
          </div>
        </div>
      </div>
    <?php endforeach; ?>
  </div>

  <!-- Global Data Table -->
  <div class="card shadow-sm mt-4">
    <div class="card-body table-responsive">
      <table class="table table-sm table-bordered table-hover align-middle table-sticky mb-0">
        <thead class="table-light text-center">
          <tr>
            <th>Date</th>
            <th>Sales (Actual)</th>
            <th>Sales (Declared)</th>
            <th>Purchases</th>
            <th>Payments</th>
            <th>Petty Cash</th>
            <th>Cash Gap</th>
            <th>Variance (Actual − Declared)</th>
          </tr>
        </thead>
        <tbody class="text-center">
          <?php if (empty($dateset)): ?>
            <tr><td colspan="8" class="text-muted">No data for selected filters.</td></tr>
          <?php else: ?>
            <?php foreach ($dateset as $d => $vals):
              $cash_gap_row = $vals['sales_actual'] - ($vals['payments'] + $vals['petty']);
              $variance_row = $vals['sales_actual'] - $vals['sales_declared'];
            ?>
              <tr>
                <td><?= htmlspecialchars($d) ?></td>
                <td class="text-success fw-semibold">KSh <?= fmt($vals['sales_actual']) ?></td>
                <td class="text-primary">KSh <?= fmt($vals['sales_declared']) ?></td>
                <td class="text-danger">KSh <?= fmt($vals['purchases']) ?></td>
                <td class="text-warning">KSh <?= fmt($vals['payments']) ?></td>
                <td class="text-secondary">KSh <?= fmt($vals['petty']) ?></td>
                <td class="<?= $cash_gap_row>=0?'text-success':'text-danger' ?> fw-semibold">KSh <?= fmt($cash_gap_row) ?></td>
                <td class="<?= $variance_row==0?'text-muted':($variance_row>0?'text-success':'text-danger') ?>">
                  <?= $variance_row>0?'+':'' ?>KSh <?= fmt($variance_row) ?>
                </td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
        </tbody>
        <?php if (!empty($dateset)): ?>
        <tfoot class="table-light">
          <tr class="text-center fw-bold">
            <td>Totals</td>
            <td class="text-success">KSh <?= fmt($totals['sales_actual']) ?></td>
            <td class="text-primary">KSh <?= fmt($totals['sales_declared']) ?></td>
            <td class="text-danger">KSh <?= fmt($totals['purchases']) ?></td>
            <td class="text-warning">KSh <?= fmt($totals['payments']) ?></td>
            <td class="text-secondary">KSh <?= fmt($totals['petty']) ?></td>
            <td class="<?= $cash_gap>=0?'text-success':'text-danger' ?>"><?= ($cash_gap>=0?'+':'') ?>KSh <?= fmt($cash_gap) ?></td>
            <td class="<?= ($totals['sales_actual']-$totals['sales_declared'])>=0?'text-success':'text-danger' ?>">
              <?= ($totals['sales_actual']-$totals['sales_declared'])>0?'+':'' ?>KSh <?= fmt($totals['sales_actual']-$totals['sales_declared']) ?>
            </td>
          </tr>
        </tfoot>
        <?php endif; ?>
      </table>
    </div>
  </div>

</div>

<!-- Scripts -->
<script>
/** ===== Chart.js helpers tuned for space efficiency ===== */
function makeMoneyFormatter() {
  return (value) => 'KSh ' + Number(value).toLocaleString();
}
const commonOpts = {
  responsive: true,
  maintainAspectRatio: false,      // allow canvas to fill container
  interaction: { mode: 'index', intersect: false },
  elements: {
    point: { radius: 0 },          // reduce visual noise
    line: { tension: .25 }         // smooth lines
  },
  plugins: {
    legend: { position: 'top', labels: { boxWidth: 16, usePointStyle: true } },
    tooltip: {
      callbacks: { label: (ctx) => `${ctx.dataset.label}: KSh ${Number(ctx.parsed.y).toLocaleString()}` }
    }
  },
  layout: { padding: { top: 8, left: 6, right: 6, bottom: 0 } },
  scales: {
    y: {
      beginAtZero: true,
      ticks: { callback: makeMoneyFormatter() },
      title: { display: true, text: 'KES' },
      grid: { drawBorder: false }
    },
    x: {
      ticks: { maxRotation: 0, autoSkip: true },
      grid: { display: false }
    }
  }
};

// Global chart
const labels = <?= json_encode($labels) ?>;
const salesActual = <?= json_encode($salesActualData) ?>;
const salesDeclared = <?= json_encode($salesDeclaredData) ?>;
const purchases = <?= json_encode($purchaseData) ?>;
const payments = <?= json_encode($paymentData) ?>;
const petty = <?= json_encode($pettyData) ?>;

// Stacked bars for Sales (Actual vs Declared) to save horizontal space
const ctx = document.getElementById('comboChart');
new Chart(ctx, {
  data: {
    labels: labels,
    datasets: [
      { type: 'bar', label: 'Sales (Actual)',   data: salesActual,   yAxisID: 'y', stack: 'sales' },
      { type: 'bar', label: 'Sales (Declared)', data: salesDeclared, yAxisID: 'y', stack: 'sales' },
      { type: 'line', label: 'Purchases', data: purchases, yAxisID: 'y' },
      { type: 'line', label: 'Payments',  data: payments,  yAxisID: 'y' },
      { type: 'line', label: 'Petty Cash',data: petty,     yAxisID: 'y' },
    ]
  },
  options: {
    ...commonOpts,
    scales: {
      ...commonOpts.scales,
      y: { ...commonOpts.scales.y, stacked: false },
      x: { ...commonOpts.scales.x, stacked: true } // stack only bars
    }
  }
});

// Branch payloads
const branchPayload = <?= json_encode($branchPayload) ?>;

// Small multiples (grid) — compact charts
Object.keys(branchPayload).forEach((bid) => {
  const el = document.getElementById('branchChart' + bid);
  if (!el) return;
  const bp = branchPayload[bid];
  new Chart(el, {
    data: {
      labels: bp.labels,
      datasets: [
        { type: 'bar',  label: 'Sales (Actual)',   data: bp.sales_actual,   stack: 'sales' },
        { type: 'bar',  label: 'Sales (Declared)', data: bp.sales_declared, stack: 'sales' },
        // Keep branch grid minimal for clarity + performance
        { type: 'line', label: 'Petty', data: bp.petty }
      ]
    },
    options: {
      ...commonOpts,
      plugins: {
        ...commonOpts.plugins,
        legend: { position: 'bottom', labels: { boxWidth: 12, usePointStyle: true } },
      },
      scales: {
        ...commonOpts.scales,
        x: { ...commonOpts.scales.x, stacked: true },
        y: { ...commonOpts.scales.y, stacked: false }
      }
    }
  });
});

// Branch large — full detail
Object.keys(branchPayload).forEach((bid) => {
  const el = document.getElementById('branchBigChart' + bid);
  if (!el) return;
  const bp = branchPayload[bid];
  const datasets = [
    { type: 'bar',  label: 'Sales (Actual)',   data: bp.sales_actual,   stack: 'sales' },
    { type: 'bar',  label: 'Sales (Declared)', data: bp.sales_declared, stack: 'sales' },
    { type: 'line', label: 'Petty',      data: bp.petty }
  ];
  // Only show purchases/payments per-branch if enabled (to avoid confusion)
  <?php if ($HAS_BRANCH_IN_PURCHASES): ?>
    datasets.push({ type: 'line', label: 'Purchases', data: bp.purchases });
  <?php endif; ?>
  <?php if ($HAS_BRANCH_IN_PAYMENTS): ?>
    datasets.push({ type: 'line', label: 'Payments',  data: bp.payments  });
  <?php endif; ?>

  new Chart(el, {
    data: { labels: bp.labels, datasets },
    options: {
      ...commonOpts,
      scales: {
        ...commonOpts.scales,
        x: { ...commonOpts.scales.x, stacked: true },
        y: { ...commonOpts.scales.y, stacked: false }
      }
    }
  });
});

// Handle tab shown -> resize charts inside newly visible panes
document.addEventListener('shown.bs.tab', (e) => {
  // Trigger a resize so Chart.js recalculates dimensions
  window.dispatchEvent(new Event('resize'));
});

// Optional: Fit charts tighter on first paint
window.addEventListener('load', () => {
  setTimeout(()=> window.dispatchEvent(new Event('resize')), 200);
});
</script>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<?php include '../../includes/footer.php'; ?>
</body>
</html>
