<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

$from = $_GET['from'] ?? date('Y-m-01');
$to   = $_GET['to']   ?? date('Y-m-d');

// === CHEQUE PAYMENTS ===
$sql1 = "
  SELECT id, payment_date AS date, reference_no, amount,
         'Cheque' AS type, 'System' AS source
  FROM payments 
  WHERE method = 'cheque' AND payment_date BETWEEN ? AND ?
";
$stmt1 = $conn->prepare($sql1);
if (!$stmt1) {
    die("Query 1 Prepare Error: " . $conn->error);
}
$stmt1->bind_param('ss', $from, $to);
$stmt1->execute();
$cheques = $stmt1->get_result()->fetch_all(MYSQLI_ASSOC);

// === BANK DEPOSITS ===
$sql2 = "
  SELECT id, deposit_date AS date, reference_number AS reference_no, deposit_amount AS amount,
         'Deposit' AS type, 'Bank' AS source
  FROM bank_deposits 
  WHERE deposit_date BETWEEN ? AND ?
";
$stmt2 = $conn->prepare($sql2);
if (!$stmt2) {
    die("Query 2 Prepare Error: " . $conn->error);
}
$stmt2->bind_param('ss', $from, $to);
$stmt2->execute();
$deposits = $stmt2->get_result()->fetch_all(MYSQLI_ASSOC);

// === MATCHING ===
$reconciled = [];
$unmatched  = [];

foreach ($cheques as $c) {
    $match = null;
    foreach ($deposits as $k => $d) {
        if (
            trim($c['reference_no']) === trim($d['reference_no']) &&
            abs($c['amount'] - $d['amount']) < 0.01
        ) {
            $match = $d;
            unset($deposits[$k]); // matched, remove from unmatched
            break;
        }
    }

    if ($match) {
        $reconciled[] = [
            'ref' => $c['reference_no'],
            'cheque_date' => $c['date'],
            'deposit_date' => $match['date'],
            'amount' => $c['amount'],
            'status' => 'Matched'
        ];
    } else {
        $unmatched[] = $c;
    }
}

// Remaining unmatched deposits
$unmatched = array_merge($unmatched, $deposits);
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Reconcile Cheques & Deposits</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
  <style>
    body { background: #f5f6f9; font-size: 14px; }
    .matched-row { background: #d4edda; }
    .unmatched-row { background: #fff3cd; }
    .table th, .table td { vertical-align: middle !important; }
    .header-icon { font-size: 18px; color: #0d6efd; }
  </style>
</head>
<body>
<div class="container py-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h4><i class="fas fa-scale-balanced header-icon"></i> Statement Reconciliation</h4>
    <form method="get" class="row g-2">
      <div class="col">
        <input type="date" name="from" class="form-control form-control-sm" value="<?= htmlspecialchars($from) ?>">
      </div>
      <div class="col">
        <input type="date" name="to" class="form-control form-control-sm" value="<?= htmlspecialchars($to) ?>">
      </div>
      <div class="col">
        <button class="btn btn-sm btn-primary"><i class="fas fa-filter me-1"></i>Filter</button>
      </div>
    </form>
  </div>

  <div class="alert alert-info">
    <strong>✔ Matched:</strong> <?= count($reconciled) ?> |
    <strong>⚠ Unmatched:</strong> <?= count($unmatched) ?>
  </div>

  <!-- Matched Entries -->
  <div class="card shadow-sm mb-4">
    <div class="card-header bg-success text-white fw-bold">
      ✅ Matched Cheques vs Deposits
    </div>
    <div class="card-body p-0">
      <table class="table table-bordered table-sm table-hover mb-0">
        <thead class="table-light text-center">
          <tr>
            <th>Reference</th>
            <th>Cheque Date</th>
            <th>Deposit Date</th>
            <th>Amount</th>
            <th>Status</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($reconciled as $r): ?>
            <tr class="matched-row text-center">
              <td><?= htmlspecialchars($r['ref']) ?></td>
              <td><?= $r['cheque_date'] ?></td>
              <td><?= $r['deposit_date'] ?></td>
              <td>KES <?= number_format($r['amount'], 2) ?></td>
              <td><span class="badge bg-success"><?= $r['status'] ?></span></td>
            </tr>
          <?php endforeach; ?>
          <?php if (empty($reconciled)): ?>
            <tr><td colspan="5" class="text-center text-muted">No matches found.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>

  <!-- Unmatched Entries -->
  <div class="card shadow-sm">
    <div class="card-header bg-warning fw-bold">
      ⚠ Unmatched Entries (Pending Reconciliation)
    </div>
    <div class="card-body p-0">
      <table class="table table-bordered table-sm table-hover mb-0">
        <thead class="table-light text-center">
          <tr>
            <th>Type</th>
            <th>Date</th>
            <th>Reference</th>
            <th>Amount</th>
            <th>Source</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($unmatched as $u): ?>
            <tr class="unmatched-row text-center">
              <td><?= $u['type'] ?></td>
              <td><?= $u['date'] ?></td>
              <td><?= htmlspecialchars($u['reference_no']) ?></td>
              <td>KES <?= number_format($u['amount'], 2) ?></td>
              <td><?= $u['source'] ?></td>
            </tr>
          <?php endforeach; ?>
          <?php if (empty($unmatched)): ?>
            <tr><td colspan="5" class="text-center text-muted">All items reconciled.</td></tr>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
