<?php
// FILE: modules/returns/add.php
// Purpose: Create Credit Notes / Goods Return Notes with branch selection (from branches table),
//          supplier link, invoice link, barcode/SKU lookup, dynamic line items & totals.

session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$user_id = (int)($_SESSION['user_id'] ?? 0);

// Load dropdown data
$suppliers = $conn->query("SELECT id, name FROM suppliers ORDER BY name ASC");
$branches  = $conn->query("SELECT id, branch_name FROM branches ORDER BY branch_name ASC");

// Auto-generated GRN number template
$autoReturnNoteNo = 'GRN-' . date('Ymd') . '-' . rand(100, 999);
?>
<div class="main-content">
  <div class="d-flex justify-content-between align-items-center mb-4">
    <h4 class="fw-bold text-primary">
      <i class="fas fa-file-arrow-up me-2"></i> Add Credit Note / Goods Return Note
    </h4>
    <a href="list.php" class="btn btn-outline-secondary">
      <i class="fas fa-list me-1"></i> View All
    </a>
  </div>

  <form method="POST" action="save_credit_note.php" enctype="multipart/form-data" class="card p-4 shadow bg-white border rounded-3">
    <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
    <input type="hidden" name="created_by" value="<?= (int)$user_id ?>">

    <!-- Header fields -->
    <div class="row g-3 mb-3">
      <div class="col-md-3">
        <label class="form-label fw-semibold">Branch</label>
        <select name="branch_id" id="branch_id" class="form-select" required>
          <option value="">-- Select Branch --</option>
          <?php if ($branches && $branches->num_rows): ?>
            <?php while ($b = $branches->fetch_assoc()): ?>
              <option value="<?= (int)$b['id'] ?>"><?= htmlspecialchars($b['branch_name']) ?></option>
            <?php endwhile; ?>
          <?php endif; ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label fw-semibold">Supplier</label>
        <select name="supplier_id" id="supplier_id" class="form-select" required onchange="fetchInvoices(this.value)">
          <option value="">-- Select Supplier --</option>
          <?php if ($suppliers && $suppliers->num_rows): ?>
            <?php while ($s = $suppliers->fetch_assoc()): ?>
              <option value="<?= (int)$s['id'] ?>"><?= htmlspecialchars($s['name']) ?></option>
            <?php endwhile; ?>
          <?php endif; ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label fw-semibold">Type</label>
        <select name="note_type" id="note_type" class="form-select" required onchange="handleNoteTypeChange(this.value)">
          <option value="">-- Select Type --</option>
          <option value="Credit Note">Credit Note</option>
          <option value="Goods Return Note">Goods Return Note</option>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label fw-semibold">Note Number</label>
        <input type="text" name="credit_note_number" id="credit_note_number" class="form-control" required placeholder="Enter or auto">
      </div>
    </div>

    <div class="row g-3 mb-3">
      <div class="col-md-4">
        <label class="form-label fw-semibold">Date</label>
        <input type="date" name="credit_date" class="form-control" required value="<?= date('Y-m-d') ?>">
      </div>
      <div class="col-md-4">
        <label class="form-label fw-semibold">Reason</label>
        <select name="reason" id="reason" class="form-select" required onchange="updateDescriptionNote()">
          <option value="">-- Select Reason --</option>
          <option>Damaged Goods</option>
          <option>Short Expiry</option>
          <option>Not Ordered Goods</option>
          <option>Supplier Discount</option>
          <option>Goods Not Delivered</option>
        </select>
      </div>
      <div class="col-md-4">
        <label class="form-label fw-semibold">Link to Invoice (Optional)</label>
        <select name="invoice_id" id="invoice_select" class="form-select">
          <option value="">-- None --</option>
        </select>
      </div>
    </div>

    <div class="mb-3">
      <label class="form-label fw-semibold">Auto Description Note</label>
      <textarea name="description_note" id="description_note" class="form-control bg-light" rows="2" readonly></textarea>
    </div>

    <!-- Lookup / add items -->
    <div class="row g-2 align-items-end mb-2">
      <div class="col-12 col-md-3">
        <label class="form-label">Scan/Enter Barcode</label>
        <input id="lookup_barcode" class="form-control" placeholder="Scan barcode">
      </div>
      <div class="col-12 col-md-5 position-relative">
        <label class="form-label">Search Item (SKU/Name)</label>
        <input id="lookup_q" class="form-control" placeholder="Type to search...">
        <div id="lookup_results" class="list-group position-absolute w-100" style="z-index:10;"></div>
      </div>
      <div class="col-12 col-md-2">
        <button class="btn btn-outline-primary w-100" type="button" onclick="searchItems()">Search</button>
      </div>
      <div class="col-12 col-md-2">
        <button class="btn btn-outline-secondary w-100" type="button" onclick="addRow()">+ Blank Row</button>
      </div>
    </div>

    <!-- Items table -->
    <div class="table-responsive">
      <table class="table table-bordered align-middle" id="items_tbl">
        <thead class="table-light">
          <tr>
            <th>SKU</th>
            <th>Item Name</th>
            <th>Unit</th>
            <th class="text-end">Qty</th>
            <th class="text-end">Unit Price</th>
            <th class="text-end">VAT %</th>
            <th class="text-end">Line Total</th>
            <th></th>
          </tr>
        </thead>
        <tbody></tbody>
        <tfoot>
          <tr>
            <th colspan="4"></th>
            <th class="text-end">Subtotal</th>
            <th class="text-end">VAT</th>
            <th class="text-end">Total</th>
            <th></th>
          </tr>
          <tr>
            <td colspan="4"></td>
            <td class="text-end"><span id="ft_subtotal">0.00</span></td>
            <td class="text-end"><span id="ft_vat">0.00</span></td>
            <td class="text-end"><span id="ft_total">0.00</span></td>
            <td></td>
          </tr>
        </tfoot>
      </table>
    </div>

    <div class="row g-3 mb-3">
      <div class="col-md-6">
        <label class="form-label fw-semibold">Upload Credit Note (PDF Optional)</label>
        <input type="file" name="credit_note_file" class="form-control" accept=".pdf">
      </div>
      <div class="col-md-6">
        <label class="form-label fw-semibold">Amount (KES)</label>
        <input type="number" name="amount" id="amount" step="0.01" class="form-control" required readonly>
      </div>
    </div>

    <div class="text-end">
      <button class="btn btn-primary px-4">
        <i class="fas fa-save me-1"></i> Save
      </button>
    </div>
  </form>
</div>

<script>
// Auto number for GRN
function handleNoteTypeChange(type) {
  const i = document.getElementById('credit_note_number');
  i.value = (type === 'Goods Return Note') ? '<?= $autoReturnNoteNo ?>' : '';
}

// Auto description
function updateDescriptionNote() {
  const r = document.getElementById('reason').value;
  const map = {
    'Damaged Goods': 'Returned due to physical damage during delivery or handling.',
    'Short Expiry': 'Products near expiry; returned or discounted accordingly.',
    'Not Ordered Goods': 'Items delivered were not part of the original order.',
    'Supplier Discount': 'Discount applied as per supplier agreement or promotion.',
    'Goods Not Delivered': 'Credit issued for goods invoiced but not delivered.'
  };
  document.getElementById('description_note').value = map[r] || '';
}

// Add item row (optionally with prefilled values)
function addRow(item) {
  const tb = document.querySelector('#items_tbl tbody');
  const tr = document.createElement('tr');
  tr.innerHTML = `
    <td><input name="sku[]" class="form-control form-control-sm" value="${item && item.sku ? item.sku : ''}"></td>
    <td><input name="item_name[]" class="form-control form-control-sm" value="${item && item.name ? item.name : ''}" required></td>
    <td><input name="unit[]" class="form-control form-control-sm" value="${item && item.unit ? item.unit : ''}" placeholder="pcs/carton"></td>
    <td><input name="qty[]" type="number" step="0.001" min="0" class="form-control form-control-sm text-end" value="${item && item.qty ? item.qty : ''}" oninput="recalc()" required></td>
    <td><input name="unit_price[]" type="number" step="0.0001" min="0" class="form-control form-control-sm text-end" value="${item && item.price ? item.price : ''}" oninput="recalc()" required></td>
    <td><input name="vat_rate[]" type="number" step="0.01" min="0" class="form-control form-control-sm text-end" value="${item && item.vat ? item.vat : 16}" oninput="recalc()"></td>
    <td class="text-end line-total">0.00</td>
    <td class="text-center">
      <button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); recalc();">
        <i class="fas fa-times"></i>
      </button>
    </td>
  `;
  tb.appendChild(tr);
  recalc();
}

// Recalculate totals
function recalc() {
  let sub=0, vat=0, tot=0;
  document.querySelectorAll('#items_tbl tbody tr').forEach(tr => {
    const q = parseFloat(tr.querySelector('[name="qty[]"]').value || 0);
    const p = parseFloat(tr.querySelector('[name="unit_price[]"]').value || 0);
    const r = parseFloat(tr.querySelector('[name="vat_rate[]"]').value || 0);
    const s = q * p;
    const v = s * (r/100);
    const t = s + v;
    tr.querySelector('.line-total').textContent = t.toFixed(2);
    sub += s; vat += v; tot += t;
  });
  document.getElementById('ft_subtotal').textContent = sub.toFixed(2);
  document.getElementById('ft_vat').textContent = vat.toFixed(2);
  document.getElementById('ft_total').textContent = tot.toFixed(2);
  document.getElementById('amount').value = tot.toFixed(2);
}

// Fetch supplier invoices for linking
function fetchInvoices(supplierId) {
  fetch("fetch_invoices.php?supplier_id=" + encodeURIComponent(supplierId))
    .then(r => r.json())
    .then(rows => {
      const s = document.getElementById('invoice_select');
      s.innerHTML = '<option value="">-- None --</option>';
      rows.forEach(inv => {
        const total = Number(inv.total_amount).toLocaleString('en-KE', { style:'currency', currency:'KES' });
        const bal = (Number(inv.balance)).toFixed(2);
        const o = document.createElement('option');
        o.value = inv.id;
        o.textContent = `${inv.invoice_number} (${inv.invoice_date}) - ${total} | Bal: ${bal}`;
        s.appendChild(o);
      });
    });
}

// Item search / barcode scan (search_items.php returns JSON)
async function searchItems() {
  const q  = document.getElementById('lookup_q').value.trim();
  const bc = document.getElementById('lookup_barcode').value.trim();
  if (!q && !bc) return;

  const res  = await fetch(`search_items.php?q=${encodeURIComponent(q)}&barcode=${encodeURIComponent(bc)}`);
  const data = await res.json();

  const list = document.getElementById('lookup_results');
  list.innerHTML = '';
  data.forEach(it => {
    const a = document.createElement('a');
    a.href = '#';
    a.className = 'list-group-item list-group-item-action';
    a.textContent = `${it.sku || ''} — ${it.name}`;
    a.onclick = (e) => {
      e.preventDefault();
      addRow({ sku: it.sku, name: it.name, unit: it.unit, price: it.cost_price, vat: it.vat_rate, qty: 1 });
      list.innerHTML = '';
      document.getElementById('lookup_q').value = '';
      document.getElementById('lookup_barcode').value = '';
    };
    list.appendChild(a);
  });
}

// Enter-to-search on barcode, live search on text
document.getElementById('lookup_barcode').addEventListener('keydown', (e) => {
  if (e.key === 'Enter') { e.preventDefault(); searchItems(); }
});
document.getElementById('lookup_q').addEventListener('input', () => searchItems());
</script>
