<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using calendar.php version: y1t2u3v4-0w1x-6y7z-2a3b-4c5d6e7f8g9");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/calendar_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Fetch branches
$branches = [];
try {
    $query = "SELECT id, branch_name FROM branches ORDER BY branch_name";
    $result = $conn->query($query);
    while ($row = $result->fetch_assoc()) {
        $branches[] = $row;
    }
    error_log("[$timestamp] Fetched " . count($branches) . " branches");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($branches) . " branches\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch branches failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch branches failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load branches: ' . $e->getMessage();
}

// Close connection
$conn->close();

// Include layout after logic
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Schedule Calendar - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/fullcalendar@5.11.3/main.min.css" rel="stylesheet">
    <style>
        #calendar { max-width: 1100px; margin: 0 auto; }
        .print-header { display: none; }
        .schedule-list { max-width: 1100px; margin: 20px auto; }
        .schedule-list h4 { margin-top: 20px; }
        .schedule-entry { padding: 5px 10px; border-bottom: 1px solid #ddd; }

        @media print {
            body { margin: 0; }
            .container, #calendar, .schedule-list { max-width: 100%; width: 100%; margin: 0; padding: 10mm; }
            header, footer, .btn, .form-select, .alert, .mb-4 { display: none !important; }
            .print-header {
                display: block;
                text-align: center;
                margin-bottom: 10mm;
                font-size: 16pt;
                font-weight: bold;
            }
            .fc { font-size: 12pt; }
            .fc-daygrid-day-number { font-size: 10pt; }
            .fc-event { font-size: 10pt; padding: 2px; }
            .schedule-entry { font-size: 10pt; }
            @page { size: A4; margin: 10mm; }
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Schedule Calendar</h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success"><?php echo $_SESSION['success']; unset($_SESSION['success']); ?></div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger"><?php echo $_SESSION['error']; unset($_SESSION['error']); ?></div>
        <?php endif; ?>
        <div class="print-header">
            StaffSync360 Schedule Calendar - <span id="print-branch">All Branches</span> - <span id="print-month"></span>
        </div>
        <div class="mb-4">
            <label for="branch_id" class="form-label">Branch</label>
            <select id="branch_id" class="form-select">
                <option value="0">All Branches</option>
                <?php foreach ($branches as $branch): ?>
                    <option value="<?php echo $branch['id']; ?>"><?php echo htmlspecialchars($branch['branch_name']); ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="mb-4">
            <button onclick="window.print()" class="btn btn-primary"><i class="fas fa-print"></i> Print Calendar</button>
        </div>
        <div id="calendar"></div>

        <div class="schedule-list">
            <h4>Schedule List</h4>
            <div id="schedule-entries" class="border rounded p-2">
                <p class="text-muted">Loading schedule...</p>
            </div>
        </div>
    </div>
    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/fullcalendar@5.11.3/main.min.js"></script>
    <script>
        let allEvents = [];
        document.addEventListener('DOMContentLoaded', function() {
            var calendarEl = document.getElementById('calendar');
            var calendar = new FullCalendar.Calendar(calendarEl, {
                initialView: 'dayGridMonth',
                events: function(fetchInfo, successCallback, failureCallback) {
                    var branchId = document.getElementById('branch_id').value;
                    var url = 'get_schedules.php?start=' + encodeURIComponent(fetchInfo.startStr) + 
                              '&end=' + encodeURIComponent(fetchInfo.endStr) + 
                              '&branch_id=' + encodeURIComponent(branchId);
                    fetch(url)
                        .then(response => {
                            if (!response.ok) {
                                throw new Error('HTTP error ' + response.status);
                            }
                            return response.json();
                        })
                        .then(data => {
                            allEvents = data;
                            updateScheduleList();
                            successCallback(data);
                        })
                        .catch(error => {
                            console.error('Error fetching events:', error);
                            failureCallback(error);
                        });
                },
                eventContent: function(arg) {
                    return { 
                        html: `<b>${arg.event.title}</b><br>Shift: ${arg.event.extendedProps.shift_type}`
                    };
                },
                datesSet: function(dateInfo) {
                    var monthYear = dateInfo.view.calendar.getDate().toLocaleString('default', { month: 'long', year: 'numeric' });
                    document.getElementById('print-month').textContent = monthYear;
                }
            });
            calendar.render();

            document.getElementById('branch_id').addEventListener('change', function() {
                var branchText = this.options[this.selectedIndex].text;
                document.getElementById('print-branch').textContent = branchText;
                calendar.refetchEvents();
            });

            function updateScheduleList() {
                const container = document.getElementById('schedule-entries');
                container.innerHTML = '';
                if (allEvents.length === 0) {
                    container.innerHTML = '<p class="text-muted">No schedules available for this selection.</p>';
                    return;
                }

                // Group events by date
                const grouped = {};
                allEvents.forEach(ev => {
                    const date = ev.start.split('T')[0];
                    if (!grouped[date]) grouped[date] = [];
                    grouped[date].push(ev);
                });

                Object.keys(grouped).sort().forEach(date => {
                    const dateHeader = document.createElement('h5');
                    dateHeader.textContent = new Date(date).toLocaleDateString(undefined, { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' });
                    container.appendChild(dateHeader);

                    grouped[date].forEach(ev => {
                        const entry = document.createElement('div');
                        entry.className = 'schedule-entry';
                        entry.innerHTML = `<b>${ev.title}</b> — Shift: ${ev.extendedProps.shift_type || 'N/A'}`;
                        container.appendChild(entry);
                    });
                });
            }
        });
    </script>
</body>
</html>
