<?php
session_start();
require_once '../../config/db.php';

$timestamp = date('Y-m-d H:i:s');
$log_dir = '../logs';
$log_file = "$log_dir/view_debug.txt";
if (!is_dir($log_dir)) mkdir($log_dir, 0755, true);

$national_id = $_GET['national_id'] ?? '';
file_put_contents($log_file, "[$timestamp] national_id requested: $national_id\n", FILE_APPEND);

if (empty($national_id)) {
    $_SESSION['error'] = 'No National ID provided.';
    header('Location: list.php');
    exit;
}

// Fetch staff details using national_id
$staff = null;
try {
    $query = "
        SELECT s.national_id, s.first_name, s.last_name, s.email, s.role, s.photo, b.branch_name
        FROM staff s
        JOIN branches b ON s.branch_id = b.id
        WHERE s.national_id = ?
    ";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('s', $national_id);
    $stmt->execute();
    $staff = $stmt->get_result()->fetch_assoc();
    if (!$staff) throw new Exception("Staff not found: $national_id");
} catch (Exception $e) {
    $_SESSION['error'] = 'Failed to load staff details: ' . $e->getMessage();
    header('Location: list.php');
    exit;
}

// Recent schedules (last 30 days)
$schedules = [];
try {
    $query = "
        SELECT schedule_date, shift_type, reason
        FROM schedules
        WHERE staff_id = ? AND schedule_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        ORDER BY schedule_date DESC
        LIMIT 10
    ";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('s', $national_id);
    $stmt->execute();
    $schedules = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
} catch (Exception $e) {
    $_SESSION['error'] = 'Failed to load schedules.';
}

// Recent attendance (last 30 days)
$attendance = [];
try {
    $query = "
        SELECT attendance_date, status
        FROM attendance
        WHERE national_id = ? AND attendance_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        ORDER BY attendance_date DESC
        LIMIT 10
    ";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('s', $national_id);
    $stmt->execute();
    $attendance = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
} catch (Exception $e) {
    $_SESSION['error'] = 'Failed to load attendance.';
}

$conn->close();
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>View Staff - StaffSync360</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <style>
    .card { border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
    .card-header { background-color: #007bff; color: white; border-radius: 10px 10px 0 0; }
    .status-absent { background-color: #f8d7da; color: #721c24; }
    .status-present { background-color: #d4edda; color: #155724; }
    .img-preview {
      border-radius: 10px;
      width: 250px;
      height: auto;
      border: 2px solid #ccc;
      box-shadow: 2px 2px 8px rgba(0,0,0,0.2);
    }
    .dl-horizontal dt { float: left; width: 160px; clear: left; text-align: right; font-weight: bold; }
    .dl-horizontal dd { margin-left: 180px; margin-bottom: 10px; }
    @media (max-width: 576px) {
      .dl-horizontal dt { float: none; width: auto; text-align: left; }
      .dl-horizontal dd { margin-left: 0; }
    }
  </style>
</head>
<body>
<div class="container mt-4">
  <h2 class="mb-4">View Staff Details</h2>

  <?php if (isset($_SESSION['error'])): ?>
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
      <?= $_SESSION['error']; unset($_SESSION['error']); ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <?php if ($staff): ?>
    <div class="card mb-4">
      <div class="card-header">
        <h5 class="card-title mb-0">Staff Information</h5>
      </div>
      <div class="card-body">
        <div class="row">
          <div class="col-md-7">
            <dl class="dl-horizontal">
              <dt>National ID</dt><dd><?= htmlspecialchars($staff['national_id']); ?></dd>
              <dt>Name</dt><dd><?= htmlspecialchars($staff['first_name'] . ' ' . $staff['last_name']); ?></dd>
              <dt>Email</dt><dd><?= htmlspecialchars($staff['email'] ?: 'N/A'); ?></dd>
              <dt>Role</dt><dd><?= htmlspecialchars($staff['role']); ?></dd>
              <dt>Branch</dt><dd><?= htmlspecialchars($staff['branch_name']); ?></dd>
            </dl>
          </div>
          <div class="col-md-5 text-center">
            <?php
            $photoPath = "../../uploads/staff_photos/nid_" . $staff['national_id'] . ".jpg";
            if (!empty($staff['photo']) && file_exists($photoPath)) {
              echo '<img src="' . $photoPath . '?v=' . time() . '" alt="Staff Photo" class="img-thumbnail img-preview">';
            } else {
              echo '<div class="text-muted">No photo uploaded</div>';
            }
            ?>
          </div>
        </div>
        <div class="mt-3">
          <a href="list.php" class="btn btn-primary"><i class="fas fa-arrow-left"></i> Back to List</a>
          <a href="edit.php?national_id=<?= urlencode($staff['national_id']); ?>" class="btn btn-secondary"><i class="fas fa-edit"></i> Edit</a>
        </div>
      </div>
    </div>

    <div class="card mb-4">
      <div class="card-header">
        <h5 class="card-title mb-0">Recent Schedules (Last 30 Days)</h5>
      </div>
      <div class="card-body">
        <?php if (empty($schedules)): ?>
          <p>No schedules found.</p>
        <?php else: ?>
          <table class="table table-striped table-hover">
            <thead><tr><th>Date</th><th>Shift Type</th><th>Reason</th></tr></thead>
            <tbody>
              <?php foreach ($schedules as $s): ?>
                <tr class="<?= $s['shift_type'] === 'off' ? 'status-absent' : ''; ?>">
                  <td><?= htmlspecialchars($s['schedule_date']); ?></td>
                  <td><?= htmlspecialchars($s['shift_type']); ?></td>
                  <td><?= htmlspecialchars($s['reason'] ?: 'N/A'); ?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        <?php endif; ?>
      </div>
    </div>

    <div class="card mb-4">
      <div class="card-header">
        <h5 class="card-title mb-0">Recent Attendance (Last 30 Days)</h5>
      </div>
      <div class="card-body">
        <?php if (empty($attendance)): ?>
          <p>No attendance records found.</p>
        <?php else: ?>
          <table class="table table-striped table-hover">
            <thead><tr><th>Date</th><th>Status</th></tr></thead>
            <tbody>
              <?php foreach ($attendance as $record): ?>
                <tr class="<?= $record['status'] === 'absent' ? 'status-absent' : 'status-present'; ?>">
                  <td><?= htmlspecialchars($record['attendance_date']); ?></td>
                  <td><?= htmlspecialchars($record['status']); ?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        <?php endif; ?>
      </div>
    </div>
  <?php endif; ?>
</div>

<?php include '../../includes/footer.php'; ?>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
