<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using muster_roll.php version: b4w5x6y7-3z4a-9b0c-5d6e-7f8g9h0i1j2-v2");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

$marked_by = $_SESSION['user_id'];

// Handle bulk absent update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_absent'])) {
    $staff_ids = $_POST['staff_ids'] ?? [];
    $date = $_POST['date'] ?? date('Y-m-d');
    $branch_id = (int)($_POST['branch_id'] ?? 0);

    if (empty($staff_ids)) {
        $_SESSION['error'] = 'No staff selected.';
        header("Location: muster_roll.php?date=" . urlencode($date) . "&branch_id=$branch_id");
        exit;
    }

    $conn->begin_transaction();
    $inserted = 0;
    try {
        foreach ($staff_ids as $staff_id) {
            $query = "
                INSERT INTO attendance (staff_id, branch_id, attendance_date, status, marked_by, marked_at)
                SELECT staff_id, branch_id, ?, 'absent', ?, NOW()
                FROM staff WHERE staff_id = ?
                ON DUPLICATE KEY UPDATE status = 'absent', marked_by = ?, marked_at = NOW()
            ";
            $stmt = $conn->prepare($query);
            $stmt->bind_param('siss', $date, $marked_by, $staff_id, $marked_by);
            $stmt->execute();
            $inserted += $stmt->affected_rows;
        }
        $conn->commit();
        $_SESSION['success'] = "Marked $inserted staff as absent.";
        error_log("[$timestamp] Bulk absent: $inserted records affected for date $date by user $marked_by");
    } catch (Exception $e) {
        $conn->rollback();
        error_log("[$timestamp] Bulk absent failed: " . $e->getMessage());
        $_SESSION['error'] = 'Failed to mark absent: ' . $e->getMessage();
    }
    header("Location: muster_roll.php?date=" . urlencode($date) . "&branch_id=$branch_id");
    exit;
}

// Handle bulk present update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_present'])) {
    $staff_ids = $_POST['staff_ids'] ?? [];
    $date = $_POST['date'] ?? date('Y-m-d');
    $branch_id = (int)($_POST['branch_id'] ?? 0);

    if (empty($staff_ids)) {
        $_SESSION['error'] = 'No staff selected.';
        header("Location: muster_roll.php?date=" . urlencode($date) . "&branch_id=$branch_id");
        exit;
    }

    $conn->begin_transaction();
    $inserted = 0;
    try {
        foreach ($staff_ids as $staff_id) {
            $query = "
                INSERT INTO attendance (staff_id, branch_id, attendance_date, status, marked_by, marked_at)
                SELECT staff_id, branch_id, ?, 'present', ?, NOW()
                FROM staff WHERE staff_id = ?
                ON DUPLICATE KEY UPDATE status = 'present', marked_by = ?, marked_at = NOW()
            ";
            $stmt = $conn->prepare($query);
            $stmt->bind_param('siss', $date, $marked_by, $staff_id, $marked_by);
            $stmt->execute();
            $inserted += $stmt->affected_rows;
        }
        $conn->commit();
        $_SESSION['success'] = "Marked $inserted staff as present.";
        error_log("[$timestamp] Bulk present: $inserted records affected for date $date by user $marked_by");
    } catch (Exception $e) {
        $conn->rollback();
        error_log("[$timestamp] Bulk present failed: " . $e->getMessage());
        $_SESSION['error'] = 'Failed to mark present: ' . $e->getMessage();
    }
    header("Location: muster_roll.php?date=" . urlencode($date) . "&branch_id=$branch_id");
    exit;
}

// Get filters
$branch_id = isset($_GET['branch_id']) ? (int)$_GET['branch_id'] : 0;
$date = $_GET['date'] ?? date('Y-m-d');
$show_absent_only = isset($_GET['show_absent_only']) && $_GET['show_absent_only'] === '1';
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$per_page = 20;

// Fetch branches
$branches = [];
try {
    $query = "SELECT id, branch_name FROM branches ORDER BY branch_name";
    $result = $conn->query($query);
    while ($row = $result->fetch_assoc()) {
        $branches[] = $row;
    }
} catch (Exception $e) {
    error_log("[$timestamp] Fetch branches failed: " . $e->getMessage());
    $_SESSION['error'] = 'Failed to load branches.';
}

// Fetch total staff count for pagination
$total_staff = 0;
try {
    $count_query = "
        SELECT COUNT(*) as total
        FROM staff s
        JOIN branches b ON s.branch_id = b.id
        LEFT JOIN attendance att ON s.staff_id = att.staff_id AND att.attendance_date = ?
        WHERE (? = 0 OR s.branch_id = ?)
        " . ($show_absent_only ? "AND (att.status = 'absent' OR att.status IS NULL)" : "");
    $count_stmt = $conn->prepare($count_query);
    $count_stmt->bind_param('sii', $date, $branch_id, $branch_id);
    $count_stmt->execute();
    $total_staff = $count_stmt->get_result()->fetch_assoc()['total'];
} catch (Exception $e) {
    error_log("[$timestamp] Count staff failed: " . $e->getMessage());
}

// Fetch muster roll with pagination
$staff = [];
$offset = ($page - 1) * $per_page;
try {
    $query = "
        SELECT s.staff_id, s.first_name, s.last_name, s.national_id, s.role, b.branch_name,
               sch.schedule_date, sch.shift_type, att.status
        FROM staff s
        JOIN branches b ON s.branch_id = b.id
        LEFT JOIN schedules sch ON s.staff_id = sch.staff_id AND sch.schedule_date = ?
        LEFT JOIN attendance att ON s.staff_id = att.staff_id AND att.attendance_date = ?
        WHERE (? = 0 OR s.branch_id = ?)
        " . ($show_absent_only ? "AND (att.status = 'absent' OR att.status IS NULL)" : "") . "
        ORDER BY s.first_name, s.last_name
        LIMIT ? OFFSET ?
    ";
    $stmt = $conn->prepare($query);
    $stmt->bind_param('ssiiss', $date, $date, $branch_id, $branch_id, $per_page, $offset);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $staff[] = $row;
    }
} catch (Exception $e) {
    error_log("[$timestamp] Fetch muster roll failed: " . $e->getMessage());
    $_SESSION['error'] = 'Failed to load muster roll.';
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Muster Roll - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .card { border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .table { font-size: 0.9rem; }
        .status-absent { background-color: #f8d7da; color: #721c24; }
        .status-present { background-color: #d4edda; color: #155724; }
        .btn-primary { background-color: #007bff; border-color: #007bff; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; }
        .btn-success { background-color: #28a745; border-color: #28a745; }
        .btn-success:hover { background-color: #218838; border-color: #218838; }
        .form-check-input:checked { background-color: #007bff; border-color: #007bff; }
        .search-bar { max-width: 300px; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Muster Roll</h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-3">
                        <label for="branch_id" class="form-label">Branch</label>
                        <select name="branch_id" id="branch_id" class="form-select">
                            <option value="0">All Branches</option>
                            <?php foreach ($branches as $branch): ?>
                                <option value="<?php echo $branch['id']; ?>" <?php echo $branch_id == $branch['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($branch['branch_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="date" class="form-label">Date</label>
                        <input type="date" name="date" id="date" class="form-control" value="<?php echo htmlspecialchars($date); ?>">
                    </div>
                    <div class="col-md-3">
                        <label for="search" class="form-label">Search Staff</label>
                        <input type="text" id="search" class="form-control search-bar" placeholder="Search by name or ID">
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <div class="form-check me-3">
                            <input class="form-check-input" type="checkbox" name="show_absent_only" id="show_absent_only" value="1" <?php echo $show_absent_only ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="show_absent_only">Show Absent Only</label>
                        </div>
                        <button type="submit" class="btn btn-primary">Filter</button>
                    </div>
                </form>
            </div>
        </div>
        <div class="card">
            <div class="card-body">
                <form method="POST" id="attendanceForm">
                    <input type="hidden" name="date" value="<?php echo htmlspecialchars($date); ?>">
                    <input type="hidden" name="branch_id" value="<?php echo $branch_id; ?>">
                    <div class="mb-3">
                        <button type="submit" name="bulk_present" class="btn btn-success me-2" onclick="return checkSelection('bulkPresentModal')">
                            <i class="fas fa-user-check"></i> Mark Selected Present
                        </button>
                        <button type="submit" name="bulk_absent" class="btn btn-danger me-2" onclick="return checkSelection('bulkAbsentModal')">
                            <i class="fas fa-user-times"></i> Mark Selected Absent
                        </button>
                        <button type="button" class="btn btn-secondary" onclick="exportToCSV()">
                            <i class="fas fa-download"></i> Export to CSV
                        </button>
                    </div>
                    <table class="table table-striped table-hover" id="staffTable">
                        <thead>
                            <tr>
                                <th><input type="checkbox" id="selectAll"></th>
                                <th>Staff ID</th>
                                <th>Name</th>
                                <th>National ID</th>
                                <th>Role</th>
                                <th>Branch</th>
                                <th>Date</th>
                                <th>Shift</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($staff as $row): ?>
                                <tr class="<?php echo $row['status'] === 'absent' ? 'status-absent' : ($row['status'] === 'present' ? 'status-present' : ''); ?>">
                                    <td><input type="checkbox" name="staff_ids[]" value="<?php echo $row['staff_id']; ?>"></td>
                                    <td><?php echo htmlspecialchars($row['staff_id']); ?></td>
                                    <td><?php echo htmlspecialchars($row['first_name'] . ' ' . $row['last_name']); ?></td>
                                    <td><?php echo htmlspecialchars($row['national_id']); ?></td>
                                    <td><?php echo htmlspecialchars($row['role']); ?></td>
                                    <td><?php echo htmlspecialchars($row['branch_name']); ?></td>
                                    <td><?php echo htmlspecialchars($row['schedule_date'] ?? $date); ?></td>
                                    <td><?php echo htmlspecialchars($row['shift_type'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($row['status'] ?? 'N/A'); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </form>
                <!-- Pagination -->
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center">
                        <?php
                        $total_pages = ceil($total_staff / $per_page);
                        for ($i = 1; $i <= $total_pages; $i++): ?>
                            <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                <a class="page-link" href="?branch_id=<?php echo $branch_id; ?>&date=<?php echo urlencode($date); ?>&show_absent_only=<?php echo $show_absent_only ? '1' : '0'; ?>&page=<?php echo $i; ?>">
                                    <?php echo $i; ?>
                                </a>
                            </li>
                        <?php endfor; ?>
                    </ul>
                </nav>
            </div>
        </div>
    </div>

    <!-- Bulk Present Modal -->
    <div class="modal fade" id="bulkPresentModal" tabindex="-1" aria-labelledby="bulkPresentModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="bulkPresentModalLabel">Confirm Bulk Present</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    Are you sure you want to mark the selected staff as present for <?php echo htmlspecialchars($date); ?>?
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-success" onclick="document.getElementById('attendanceForm').submit()">Confirm</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Bulk Absent Modal -->
    <div class="modal fade" id="bulkAbsentModal" tabindex="-1" aria-labelledby="bulkAbsentModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="bulkAbsentModalLabel">Confirm Bulk Absent</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    Are you sure you want to mark the selected staff as absent for <?php echo htmlspecialchars($date); ?>?
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger" onclick="document.getElementById('attendanceForm').submit()">Confirm</button>
                </div>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Select All Checkbox
        document.getElementById('selectAll').addEventListener('change', function() {
            var checkboxes = document.querySelectorAll('input[name="staff_ids[]"]');
            checkboxes.forEach(cb => cb.checked = this.checked);
        });

        // Check Selection for Bulk Action
        function checkSelection(modalId) {
            var checkboxes = document.querySelectorAll('input[name="staff_ids[]"]:checked');
            if (checkboxes.length === 0) {
                alert('Please select at least one staff member.');
                return false;
            }
            var modal = new bootstrap.Modal(document.getElementById(modalId));
            modal.show();
            return false;
        }

        // Search Bar
        document.getElementById('search').addEventListener('input', function() {
            var search = this.value.toLowerCase();
            var rows = document.querySelectorAll('#staffTable tbody tr');
            rows.forEach(row => {
                var staffId = row.cells[1].textContent.toLowerCase();
                var name = row.cells[2].textContent.toLowerCase();
                row.style.display = (staffId.includes(search) || name.includes(search)) ? '' : 'none';
            });
        });

        // Export to CSV
        function exportToCSV() {
            var rows = document.querySelectorAll('#staffTable tr');
            var csv = [];
            rows.forEach(row => {
                var cols = row.querySelectorAll('td, th');
                var rowData = [];
                cols.forEach((col, index) => {
                    if (index !== 0) {
                        rowData.push('"' + col.textContent.replace(/"/g, '""') + '"');
                    }
                });
                csv.push(rowData.join(','));
            });
            var csvContent = 'data:text/csv;charset=utf-8,Staff ID,Name,National ID,Role,Branch,Date,Shift,Status\n' + csv.join('\n');
            var link = document.createElement('a');
            link.setAttribute('href', encodeURI(csvContent));
            link.setAttribute('download', 'muster_roll_<?php echo $date; ?>.csv');
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
    </script>
</body>
</html>