<?php
session_start();
require_once '../../config/db.php';

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/edit_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $national_id = $_POST['national_id'] ?? '';
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $email = !empty($_POST['email']) ? trim($_POST['email']) : null;
    $phone = !empty($_POST['phone']) ? trim($_POST['phone']) : null;
    $branch_id = (int)($_POST['branch_id'] ?? 0);
    $role = $_POST['role'] ?? '';
    $status = $_POST['status'] ?? '';

    // Validate national_id
    if (!preg_match('/^\d{7,9}$/', $national_id)) {
        $_SESSION['error'] = 'Invalid National ID format.';
        error_log("[$timestamp] Invalid national_id in save: $national_id");
        header("Location: edit.php?national_id=" . urlencode($national_id));
        exit;
    }

    // Validate required fields
    if (!preg_match('/^[A-Za-z\s]{1,50}$/', $first_name)) {
        $_SESSION['error'] = 'Invalid first name. Use 1-50 letters or spaces.';
        error_log("[$timestamp] Invalid first_name for national_id: $national_id");
        header("Location: edit.php?national_id=" . urlencode($national_id));
        exit;
    }

    if (!preg_match('/^[A-Za-z\s]{1,50}$/', $last_name)) {
        $_SESSION['error'] = 'Invalid last name. Use 1-50 letters or spaces.';
        error_log("[$timestamp] Invalid last_name for national_id: $national_id");
        header("Location: edit.php?national_id=" . urlencode($national_id));
        exit;
    }

    // Validate optional fields
    if ($email && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $_SESSION['error'] = 'Invalid email address.';
        error_log("[$timestamp] Invalid email for national_id: $national_id");
        header("Location: edit.php?national_id=" . urlencode($national_id));
        exit;
    }

    if ($phone && !preg_match('/^\+?[0-9]{10,15}$/', $phone)) {
        $_SESSION['error'] = 'Invalid phone number. Use 10-15 digits, optionally starting with +.';
        error_log("[$timestamp] Invalid phone for national_id: $national_id");
        header("Location: edit.php?national_id=" . urlencode($national_id));
        exit;
    }

    // Validate branch_id
    $branch_query = "SELECT id FROM branches WHERE id = ?";
    $branch_stmt = $conn->prepare($branch_query);
    $branch_stmt->bind_param('i', $branch_id);
    $branch_stmt->execute();
    if ($branch_stmt->get_result()->num_rows === 0) {
        $_SESSION['error'] = 'Invalid branch selected.';
        error_log("[$timestamp] Invalid branch_id: $branch_id for national_id: $national_id");
        header("Location: edit.php?national_id=" . urlencode($national_id));
        exit;
    }

    // Validate role
    $valid_roles = [
        'manager', 'supervisor', 'accountant', 'chief_cashier', 'cashier', 'line_manager',
        'store_clerk', 'security', 'baker', 'receiving_clerk', 'general', 'packer'
    ];
    if (!in_array($role, $valid_roles)) {
        $_SESSION['error'] = 'Invalid role selected.';
        error_log("[$timestamp] Invalid role: $role for national_id: $national_id");
        header("Location: edit.php?national_id=" . urlencode($national_id));
        exit;
    }

    // Validate status
    if (!in_array($status, ['active', 'inactive'])) {
        $_SESSION['error'] = 'Invalid status selected.';
        error_log("[$timestamp] Invalid status: $status for national_id: $national_id");
        header("Location: edit.php?national_id=" . urlencode($national_id));
        exit;
    }

    try {
        // Update staff record
        $query = "
            UPDATE staff
            SET first_name = ?, last_name = ?, email = ?, phone = ?, branch_id = ?, role = ?, status = ?
            WHERE national_id = ?
        ";
        $stmt = $conn->prepare($query);
        $stmt->bind_param('ssssisss', $first_name, $last_name, $email, $phone, $branch_id, $role, $status, $national_id);
        $stmt->execute();

        if ($stmt->affected_rows > 0) {
            $_SESSION['success'] = "Staff record for $first_name $last_name updated successfully.";
            error_log("[$timestamp] Staff updated for national_id: $national_id");
            file_put_contents($log_file, "[$timestamp] Staff updated for national_id: $national_id\n", FILE_APPEND);
        } else {
            $_SESSION['error'] = 'No changes made to staff record.';
            error_log("[$timestamp] No changes for national_id: $national_id");
            file_put_contents($log_file, "[$timestamp] No changes for national_id: $national_id\n", FILE_APPEND);
        }
    } catch (Exception $e) {
        $_SESSION['error'] = 'Failed to update staff: ' . $e->getMessage();
        error_log("[$timestamp] Update failed for national_id: $national_id, error: " . $e->getMessage());
        file_put_contents($log_file, "[$timestamp] Update failed for national_id: $national_id, error: " . $e->getMessage() . "\n", FILE_APPEND);
    }

    $conn->close();
    header("Location: edit.php?national_id=" . urlencode($national_id));
    exit;
} else {
    $_SESSION['error'] = 'Invalid request method.';
    error_log("[$timestamp] Invalid request method for save_staff.php");
    header('Location: list.php');
    exit;
}
?>