<?php
session_start();
require_once '../../config/db.php';

// Log script version
error_log("Using staff_payments/loan.php version: b1c2d3e4-f5a6-4b7c-8d9e-0a1b2c3d4e5f");

// Define timestamp for logging
$timestamp = date('Y-m-d H:i:s');

// Ensure logs directory exists
$log_dir = '../logs';
$log_file = "$log_dir/staff_payments_debug.txt";
if (!is_dir($log_dir)) {
    mkdir($log_dir, 0755, true);
}

// Generate CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Get staff_id
$staff_id = $_GET['staff_id'] ?? '';
if (empty($staff_id)) {
    $_SESSION['error'] = 'Invalid staff ID.';
    header("Location: list.php");
    exit;
}

// Fetch staff details
try {
    $stmt = $conn->prepare("
        SELECT s.staff_id, s.first_name, s.last_name, s.monthly_salary
        FROM staff s
        WHERE s.staff_id = ? AND s.status = 'active'
    ");
    $stmt->bind_param('s', $staff_id);
    $stmt->execute();
    $staff = $stmt->get_result()->fetch_assoc();
    if (!$staff) {
        $_SESSION['error'] = 'Staff not found or inactive.';
        header("Location: list.php");
        exit;
    }
} catch (Exception $e) {
    error_log("[$timestamp] Fetch staff failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch staff failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load staff details.';
    header("Location: list.php");
    exit;
}

// Fetch existing loans
$loans = [];
try {
    $stmt = $conn->prepare("
        SELECT loan_id, amount, monthly_deduction, status, requested_at, approved_at, approved_by
        FROM loans
        WHERE staff_id = ?
        ORDER BY requested_at DESC
    ");
    $stmt->bind_param('s', $staff_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $loans[] = $row;
    }
    error_log("[$timestamp] Fetched " . count($loans) . " loans for staff ID $staff_id");
    file_put_contents($log_file, "[$timestamp] Fetched " . count($loans) . " loans for staff ID $staff_id\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("[$timestamp] Fetch loans failed: " . $e->getMessage());
    file_put_contents($log_file, "[$timestamp] Fetch loans failed: " . $e->getMessage() . "\n", FILE_APPEND);
    $_SESSION['error'] = 'Failed to load loans.';
}

// Handle loan request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['request_loan'])) {
    if ($_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = 'Invalid CSRF token.';
        header("Location: loan.php?staff_id=" . urlencode($staff_id));
        exit;
    }

    $amount = (float)$_POST['amount'];
    $repayment_months = (int)$_POST['repayment_months'];

    // Validate inputs
    $max_loan = $staff['monthly_salary'] * 3;
    if ($amount <= 0 || $amount > $max_loan || $repayment_months < 6 || $repayment_months > 12) {
        $_SESSION['error'] = "Invalid loan details. Amount must be ≤ KSh " . number_format($max_loan, 2) . " and repayment 6-12 months.";
        header("Location: loan.php?staff_id=" . urlencode($staff_id));
        exit;
    }

    $monthly_deduction = $amount / $repayment_months;

    try {
        $stmt = $conn->prepare("
            INSERT INTO loans (staff_id, amount, monthly_deduction, status)
            VALUES (?, ?, ?, 'pending')
        ");
        $stmt->bind_param('sdd', $staff_id, $amount, $monthly_deduction);
        $stmt->execute();
        error_log("[$timestamp] Loan request of KSh $amount for staff ID $staff_id");
        file_put_contents($log_file, "[$timestamp] Loan request of KSh $amount for staff ID $staff_id\n", FILE_APPEND);
        $_SESSION['success'] = 'Loan request submitted successfully.';
        header("Location: loan.php?staff_id=" . urlencode($staff_id));
        exit;
    } catch (Exception $e) {
        error_log("[$timestamp] Loan request failed: " . $e->getMessage());
        file_put_contents($log_file, "[$timestamp] Loan request failed: " . $e->getMessage() . "\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to submit loan request: ' . $e->getMessage();
        header("Location: loan.php?staff_id=" . urlencode($staff_id));
        exit;
    }
}

// Handle loan approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['approve_loan'])) {
    if ($_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $_SESSION['error'] = 'Invalid CSRF token.';
        header("Location: loan.php?staff_id=" . urlencode($staff_id));
        exit;
    }

    $loan_id = (int)$_POST['loan_id'];
    $action = $_POST['action'];
    $status = $action === 'approve' ? 'approved' : 'rejected';
    $approved_by = $_SESSION['user_id'];

    try {
        $stmt = $conn->prepare("
            UPDATE loans
            SET status = ?, approved_by = ?, approved_at = NOW()
            WHERE loan_id = ? AND staff_id = ? AND status = 'pending'
        ");
        $stmt->bind_param('ssis', $status, $approved_by, $loan_id, $staff_id);
        $stmt->execute();
        if ($stmt->affected_rows === 0) {
            $_SESSION['error'] = 'Loan not found or already processed.';
        } else {
            error_log("[$timestamp] Loan ID $loan_id $status for staff ID $staff_id");
            file_put_contents($log_file, "[$timestamp] Loan ID $loan_id $status for staff ID $staff_id\n", FILE_APPEND);
            $_SESSION['success'] = "Loan $status successfully.";
        }
        header("Location: loan.php?staff_id=" . urlencode($staff_id));
        exit;
    } catch (Exception $e) {
        error_log("[$timestamp] Loan approval failed: " . $e->getMessage());
        file_put_contents($log_file, "[$timestamp] Loan approval failed: " . $e->getMessage() . "\n", FILE_APPEND);
        $_SESSION['error'] = 'Failed to process loan: ' . $e->getMessage();
        header("Location: loan.php?staff_id=" . urlencode($staff_id));
        exit;
    }
}

// Close connection
$conn->close();

// Include layout
include '../../includes/layout2.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Loans - StaffSync360</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); border: 1px solid #e0e0e0; }
        .btn-primary { background-color: #007bff; border-color: #007bff; transition: all 0.3s ease; }
        .btn-primary:hover { background-color: #0056b3; border-color: #0056b3; box-shadow: 0 2px 8px rgba(0,0,0,0.2); }
        .btn-success { background-color: #28a745; border-color: #28a745; }
        .btn-success:hover { background-color: #218838; border-color: #218838; }
        .btn-danger { background-color: #dc3545; border-color: #dc3545; }
        .btn-danger:hover { background-color: #c82333; border-color: #c82333; }
        .form-label { font-weight: 500; color: #333; margin-bottom: 8px; }
        .form-control, .form-select { border: 1px solid #007bff; border-radius: 5px; transition: border-color 0.3s ease; }
        .form-control:hover, .form-select:hover { border-color: #0056b3; }
        .alert { border-radius: 8px; }
        .table { font-size: 0.9rem; }
        .table-responsive { max-height: 300px; overflow-y: auto; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h2 class="mb-4">Manage Loans for <?php echo htmlspecialchars($staff['first_name'] . ' ' . $staff['last_name']); ?></h2>
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <div class="card mb-4">
            <div class="card-body">
                <h5>Request New Loan</h5>
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    <div class="mb-3">
                        <label for="amount" class="form-label">Loan Amount (KES)</label>
                        <input type="number" step="0.01" name="amount" id="amount" class="form-control" required>
                        <small class="form-text text-muted">Max: KSh <?php echo number_format($staff['monthly_salary'] * 3, 2); ?></small>
                    </div>
                    <div class="mb-3">
                        <label for="repayment_months" class="form-label">Repayment Period (Months)</label>
                        <select name="repayment_months" id="repayment_months" class="form-select" required>
                            <?php for ($i = 6; $i <= 12; $i++): ?>
                                <option value="<?php echo $i; ?>"><?php echo $i; ?> Months</option>
                            <?php endfor; ?>
                        </select>
                    </div>
                    <button type="submit" name="request_loan" class="btn btn-primary"><i class="fas fa-hand-holding-usd"></i> Request Loan</button>
                    <a href="list.php" class="btn btn-secondary">Cancel</a>
                </form>
            </div>
        </div>
        <div class="card">
            <div class="card-body">
                <h5>Existing Loans</h5>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>Loan ID</th>
                                <th>Amount (KES)</th>
                                <th>Monthly Deduction (KES)</th>
                                <th>Status</th>
                                <th>Requested At</th>
                                <th>Approved At</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($loans as $loan): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($loan['loan_id']); ?></td>
                                    <td>KSh <?php echo number_format($loan['amount'], 2); ?></td>
                                    <td>KSh <?php echo number_format($loan['monthly_deduction'], 2); ?></td>
                                    <td><?php echo htmlspecialchars($loan['status']); ?></td>
                                    <td><?php echo htmlspecialchars($loan['requested_at']); ?></td>
                                    <td><?php echo htmlspecialchars($loan['approved_at'] ?? 'N/A'); ?></td>
                                    <td>
                                        <?php if ($loan['status'] === 'pending'): ?>
                                            <form method="POST" style="display:inline;">
                                                <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                                <input type="hidden" name="loan_id" value="<?php echo $loan['loan_id']; ?>">
                                                <button type="submit" name="approve_loan" value="approve" class="btn btn-sm btn-success"><i class="fas fa-check"></i> Approve</button>
                                                <button type="submit" name="approve_loan" value="reject" class="btn btn-sm btn-danger"><i class="fas fa-times"></i> Reject</button>
                                            </form>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <?php include '../../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>