<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Filters
$from = $_GET['from'] ?? date('Y-m-d', strtotime('-7 days'));
$to = $_GET['to'] ?? date('Y-m-d');
$branch_filter = $_GET['branch'] ?? '';

// Branches
$allBranches = [];
$branchRes = $conn->query("SELECT id, branch_name FROM branches ORDER BY branch_name");
while ($b = $branchRes->fetch_assoc()) {
    $allBranches[$b['id']] = $b['branch_name'];
}

// Query base
$where = "WHERE h.report_date BETWEEN ? AND ?";
$params = [$from, $to];
$types = 'ss';

if ($branch_filter) {
    $where .= " AND b.branch_name = ?";
    $params[] = $branch_filter;
    $types .= 's';
}

$sql = "
    SELECT h.id AS report_id, h.report_date, h.remarks, b.id AS branch_id, b.branch_name
    FROM till_report_headers h
    JOIN branches b ON b.id = h.branch_id
    $where
    ORDER BY h.report_date DESC, b.branch_name ASC
";
$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();

$reportsByDate = [];
$totals = [];

while ($row = $result->fetch_assoc()) {
    $report_id = $row['report_id'];

    $cashRow = $conn->query("SELECT 
        SUM(drop1 + drop2 + drop3 + note_1000 + note_500 + note_200 + note_100 + note_50 + coins) AS total_cash,
        SUM(mpesa) AS mpesa_total,
        SUM(visa) AS visa_total
    FROM till_report_details WHERE report_id = $report_id")->fetch_assoc();

    $total_cash = floatval($cashRow['total_cash'] ?? 0);
    $mpesa_total = floatval($cashRow['mpesa_total'] ?? 0);
    $visa_total = floatval($cashRow['visa_total'] ?? 0);

    $banked = floatval($conn->query("SELECT SUM(deposit_amount) AS total FROM bank_deposits WHERE report_id = $report_id AND deposit_type = 'Cash'")->fetch_assoc()['total'] ?? 0);
    $cheque_total = floatval($conn->query("SELECT SUM(amount) AS total FROM till_report_cheques WHERE report_id = $report_id")->fetch_assoc()['total'] ?? 0);

    $row['cash'] = $total_cash;
    $row['mpesa'] = $mpesa_total;
    $row['visa'] = $visa_total;
    $row['cheques'] = $cheque_total;
    $row['banked'] = $banked;
    $row['remaining'] = $total_cash - $banked;

    $reportsByDate[$row['report_date']][$row['branch_id']] = $row;

    if (!isset($totals[$row['report_date']])) {
        $totals[$row['report_date']] = ['reported' => 0, 'cash' => 0, 'mpesa' => 0, 'visa' => 0, 'cheques' => 0];
    }

    $totals[$row['report_date']]['reported']++;
    $totals[$row['report_date']]['cash'] += $total_cash;
    $totals[$row['report_date']]['mpesa'] += $mpesa_total;
    $totals[$row['report_date']]['visa'] += $visa_total;
    $totals[$row['report_date']]['cheques'] += $cheque_total;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <title>Accountant Deposit Insights</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
  <style>
    body { background: #f4f6f9; font-size: 13px; }
    .summary-card { background: #fff; border-radius: 10px; padding: 15px; box-shadow: 0 0 5px rgba(0,0,0,0.1); margin-bottom: 15px; }
    .summary-header { font-weight: bold; background: #0d6efd; color: white; padding: 10px; border-radius: 5px 5px 0 0; }
    .remaining-ok { background-color: #d4edda; }
    .remaining-warn { background-color: #fff3cd; }
    .remaining-alert { background-color: #f8d7da; }
    .progress-bar-label { font-size: 12px; font-weight: bold; }
    .branch-label { font-weight: 600; }
  </style>
</head>
<body>
<div class="container py-4">
  <h3 class="mb-3">📘 Accountant Insights – Till Report Analysis (No Charts)</h3>

  <form class="row g-2 mb-3" method="GET">
    <div class="col-md-3"><label>From</label><input type="date" name="from" class="form-control" value="<?= htmlspecialchars($from) ?>"></div>
    <div class="col-md-3"><label>To</label><input type="date" name="to" class="form-control" value="<?= htmlspecialchars($to) ?>"></div>
    <div class="col-md-3">
      <label>Branch</label>
      <select name="branch" class="form-select">
        <option value="">All Branches</option>
        <?php foreach ($allBranches as $id => $name): ?>
          <option value="<?= $name ?>" <?= $branch_filter === $name ? 'selected' : '' ?>><?= $name ?></option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="col-md-3 d-flex align-items-end">
      <button class="btn btn-primary w-100">🔍 Apply Filter</button>
    </div>
  </form>

  <?php if (empty($reportsByDate)): ?>
    <div class="alert alert-info">No report data found for the selected period.</div>
  <?php else: ?>
    <?php foreach ($reportsByDate as $date => $branches): ?>
      <?php $summary = $totals[$date]; $missed = count($allBranches) - $summary['reported']; ?>
      <div class="summary-card">
        <div class="summary-header">
          📅 <?= $date ?> — ✅ <?= $summary['reported'] ?> Reports | ❌ <?= $missed ?> Missed |
          💵 <?= number_format($summary['cash'], 2) ?> | 📲 <?= number_format($summary['mpesa'], 2) ?> | 💳 <?= number_format($summary['visa'], 2) ?> | 🧾 <?= number_format($summary['cheques'], 2) ?>
        </div>
        <div class="table-responsive mt-2">
          <table class="table table-sm table-bordered align-middle mb-0">
            <thead class="table-light">
              <tr>
                <th>Branch</th>
                <th>Report ID</th>
                <th>Cash</th>
                <th>Banked</th>
                <th>Remaining</th>
                <th>Deposit %</th>
                <th>MPESA</th>
                <th>VISA</th>
                <th>Cheques</th>
                <th>Action</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($allBranches as $branchId => $branchName): ?>
                <?php if (isset($branches[$branchId])): $r = $branches[$branchId]; ?>
                  <?php
                    $remainingClass = $r['remaining'] == 0 ? 'remaining-ok' : ($r['remaining'] < ($r['cash'] * 0.25) ? 'remaining-warn' : 'remaining-alert');
                    $depositPercent = $r['cash'] > 0 ? round(($r['banked'] / $r['cash']) * 100) : 0;
                  ?>
                  <tr class="<?= $remainingClass ?>">
                    <td class="branch-label"><?= $branchName ?></td>
                    <td>#<?= $r['report_id'] ?></td>
                    <td>Ksh <?= number_format($r['cash'], 2) ?></td>
                    <td>Ksh <?= number_format($r['banked'], 2) ?></td>
                    <td><strong>Ksh <?= number_format($r['remaining'], 2) ?></strong></td>
                    <td>
                      <div class="progress" style="height: 15px;">
                        <div class="progress-bar bg-success" style="width: <?= $depositPercent ?>%;">
                          <span class="progress-bar-label"><?= $depositPercent ?>%</span>
                        </div>
                      </div>
                    </td>
                    <td>Ksh <?= number_format($r['mpesa'], 2) ?></td>
                    <td>Ksh <?= number_format($r['visa'], 2) ?></td>
                    <td>Ksh <?= number_format($r['cheques'], 2) ?></td>
                    <td><a href="record_deposit.php?report_id=<?= $r['report_id'] ?>" class="btn btn-sm btn-outline-primary">➕ Record</a></td>
                  </tr>
                <?php else: ?>
                  <tr class="table-danger">
                    <td class="text-danger fw-bold"><?= $branchName ?></td>
                    <td colspan="9">❌ No Report Submitted</td>
                  </tr>
                <?php endif; ?>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>
    <?php endforeach; ?>
  <?php endif; ?>
</div>
</body>
</html>
