<?php
require_once '../../config/db.php';
include '../../includes/layout2.php';

$from = date('Y-m-d', strtotime('-7 days'));
$to = date('Y-m-d');

$query = "
SELECT 
    b.id AS branch_id,
    b.branch_name,
    COUNT(DISTINCT h.id) AS reports,
    SUM(d.drop1 + d.drop2 + d.drop3) AS drops,
    SUM(d.note_1000*1 + d.note_500*1 + d.note_200*1 + d.note_100*1 + d.note_50*1 + d.coins) AS total_cash,
    SUM(d.mpesa) AS total_mpesa,
    SUM(d.visa) AS total_visa,
    SUM(IFNULL(dep.deposit_amount, 0)) AS deposited
FROM branches b
LEFT JOIN till_report_headers h ON h.branch_id = b.id AND h.report_date BETWEEN ? AND ?
LEFT JOIN till_report_details d ON d.report_id = h.id
LEFT JOIN bank_deposits dep ON dep.report_id = h.id
GROUP BY b.id
ORDER BY b.branch_name
";
$stmt = $conn->prepare($query);
$stmt->bind_param('ss', $from, $to);
$stmt->execute();
$result = $stmt->get_result();

$branches = [];
$grand = ['cash' => 0, 'mpesa' => 0, 'visa' => 0, 'revenue' => 0, 'deposited' => 0, 'reports' => 0];

while ($row = $result->fetch_assoc()) {
    $row['total_cash'] = $row['total_cash'] ?? 0;
    $row['total_mpesa'] = $row['total_mpesa'] ?? 0;
    $row['total_visa'] = $row['total_visa'] ?? 0;
    $row['deposited'] = $row['deposited'] ?? 0;

    $row['total'] = $row['total_cash'] + $row['total_mpesa'] + $row['total_visa'];
    $row['unbanked'] = $row['total_cash'] - $row['deposited'];
    $row['unbanked_percent'] = $row['total_cash'] > 0 ? ($row['unbanked'] / $row['total_cash']) * 100 : 0;

    $grand['cash'] += $row['total_cash'];
    $grand['mpesa'] += $row['total_mpesa'];
    $grand['visa'] += $row['total_visa'];
    $grand['revenue'] += $row['total'];
    $grand['deposited'] += $row['deposited'];
    $grand['reports'] += $row['reports'];

    $branches[] = $row;
}

$grand['unbanked'] = $grand['cash'] - $grand['deposited'];
$grand['unbanked_percent'] = $grand['cash'] > 0 ? ($grand['unbanked'] / $grand['cash']) * 100 : 0;
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Business Dashboard - SmartPay360</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css" rel="stylesheet">
  <link href="https://cdn.datatables.net/buttons/2.4.1/css/buttons.bootstrap5.min.css" rel="stylesheet">
  <style>
    body { background: #f8f9fa; font-size: 13px; }
    .card { margin-bottom: 1rem; }
    .kpi-card { padding: 10px; background: #fff; border-left: 5px solid #007bff; }
    .kpi-title { font-size: 13px; font-weight: 600; color: #555; }
    .kpi-value { font-size: 16px; font-weight: bold; }
    .table thead { background: #007bff; color: #fff; }
  </style>
</head>
<body>
<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="text-primary">📊 Business Summary Dashboard</h5>
    <div>Period: <strong><?= $from ?></strong> to <strong><?= $to ?></strong></div>
  </div>

  <div class="row g-2 mb-3">
    <div class="col-md-3"><div class="kpi-card"><div class="kpi-title">Total Cash</div><div class="kpi-value">KES <?= number_format($grand['cash'],2) ?></div></div></div>
    <div class="col-md-3"><div class="kpi-card"><div class="kpi-title">Total MPESA</div><div class="kpi-value">KES <?= number_format($grand['mpesa'],2) ?></div></div></div>
    <div class="col-md-3"><div class="kpi-card"><div class="kpi-title">Total VISA</div><div class="kpi-value">KES <?= number_format($grand['visa'],2) ?></div></div></div>
    <div class="col-md-3"><div class="kpi-card"><div class="kpi-title">Total Revenue</div><div class="kpi-value text-success">KES <?= number_format($grand['revenue'],2) ?></div></div></div>
    <div class="col-md-3"><div class="kpi-card"><div class="kpi-title">Total Deposited</div><div class="kpi-value">KES <?= number_format($grand['deposited'],2) ?></div></div></div>
    <div class="col-md-3"><div class="kpi-card"><div class="kpi-title">Unbanked Cash</div><div class="kpi-value text-danger">KES <?= number_format($grand['unbanked'],2) ?> (<?= number_format($grand['unbanked_percent'],1) ?>%)</div></div></div>
    <div class="col-md-3"><div class="kpi-card"><div class="kpi-title">Branches</div><div class="kpi-value"><?= count($branches) ?></div></div></div>
    <div class="col-md-3"><div class="kpi-card"><div class="kpi-title">Reports Submitted</div><div class="kpi-value"><?= $grand['reports'] ?></div></div></div>
  </div>

  <table class="table table-bordered table-striped table-sm" id="summaryTable">
    <thead>
      <tr>
        <th>Branch</th>
        <th>Reports</th>
        <th>Cash</th>
        <th>MPESA</th>
        <th>VISA</th>
        <th>Total</th>
        <th>Deposited</th>
        <th>Unbanked</th>
        <th>Unbanked %</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($branches as $b): ?>
        <tr>
          <td><?= htmlspecialchars($b['branch_name']) ?></td>
          <td><?= $b['reports'] ?></td>
          <td><?= number_format($b['total_cash'], 2) ?></td>
          <td><?= number_format($b['total_mpesa'], 2) ?></td>
          <td><?= number_format($b['total_visa'], 2) ?></td>
          <td class="text-success"><?= number_format($b['total'], 2) ?></td>
          <td><?= number_format($b['deposited'], 2) ?></td>
          <td><?= number_format($b['unbanked'], 2) ?></td>
          <td><?= number_format($b['unbanked_percent'], 1) ?>%</td>
        </tr>
      <?php endforeach; ?>
    </tbody>
  </table>
</div>

<script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.1/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.1/js/buttons.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.1/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.1/js/buttons.print.min.js"></script>
<script>
$(document).ready(function() {
  $('#summaryTable').DataTable({
    dom: 'Bfrtip',
    buttons: ['copy', 'excel', 'pdf', 'print'],
    pageLength: 10,
    order: [[1, 'desc']]
  });
});
</script>
</body>
</html>
