<?php
require_once '../../config/db.php';
include '../../includes/layout2.php';

if (!isset($_GET['report_id']) || !is_numeric($_GET['report_id'])) {
    echo "<div class='alert alert-danger'>Invalid report ID.</div>";
    exit;
}

$report_id = intval($_GET['report_id']);

// Fetch header info
$stmt = $conn->prepare("
    SELECT h.*, b.branch_name 
    FROM till_report_headers h 
    JOIN branches b ON b.id = h.branch_id 
    WHERE h.id = ?
");
$stmt->bind_param("i", $report_id);
$stmt->execute();
$header = $stmt->get_result()->fetch_assoc();

if (!$header) {
    echo "<div class='alert alert-danger'>Report not found.</div>";
    exit;
}

$branch_id = $header['branch_id'];
$report_date = $header['report_date'];

// Fetch till details
$stmt_details = $conn->prepare("
    SELECT * 
    FROM till_report_details 
    WHERE report_id = ? 
    ORDER BY till_no ASC
");
$stmt_details->bind_param("i", $report_id);
$stmt_details->execute();
$tills = $stmt_details->get_result()->fetch_all(MYSQLI_ASSOC);

// Fetch cheques per till
$stmt_cheques = $conn->prepare("
    SELECT till_no, customer_name, cheque_number, amount 
    FROM till_report_cheques 
    WHERE report_id = ?
    ORDER BY till_no ASC, id ASC
");
$stmt_cheques->bind_param("i", $report_id);
$stmt_cheques->execute();
$cheque_result = $stmt_cheques->get_result();

$cheques_by_till = [];
$total_cheques = 0;
while ($row = $cheque_result->fetch_assoc()) {
    $cheques_by_till[$row['till_no']][] = $row;
    $total_cheques += floatval($row['amount']);
}

// Fetch petty cash
$stmt_petty_sum = $conn->prepare("
    SELECT SUM(amount) AS total_receipts 
    FROM petty_cash_transactions 
    WHERE branch_id = ? AND transaction_date = ? 
      AND type = 'receipt' AND status = 'approved'
");
$stmt_petty_sum->bind_param("is", $branch_id, $report_date);
$stmt_petty_sum->execute();
$petty_row = $stmt_petty_sum->get_result()->fetch_assoc();
$petty_receipts = floatval($petty_row['total_receipts']);

$stmt_petty_list = $conn->prepare("
    SELECT pct.*, u.full_name 
    FROM petty_cash_transactions pct
    LEFT JOIN users u ON pct.created_by = u.id
    WHERE pct.branch_id = ? AND pct.transaction_date = ?
    ORDER BY pct.id ASC
");
$stmt_petty_list->bind_param("is", $branch_id, $report_date);
$stmt_petty_list->execute();
$petty_list = $stmt_petty_list->get_result()->fetch_all(MYSQLI_ASSOC);

$total_disbursement = $total_receipt = $total_reimbursement = $total_petty = 0;
foreach ($petty_list as $row) {
    $amt = floatval($row['amount']);
    $total_petty += $amt;
    if ($row['type'] === 'disbursement') $total_disbursement += $amt;
    if ($row['type'] === 'receipt') $total_receipt += $amt;
    if ($row['type'] === 'reimbursement') $total_reimbursement += $amt;
}

// Fetch banked
$stmt_banked = $conn->prepare("SELECT SUM(deposit_amount) AS total FROM bank_deposits WHERE report_id = ?");
$stmt_banked->bind_param("i", $report_id);
$stmt_banked->execute();
$banked_row = $stmt_banked->get_result()->fetch_assoc();
$banked = floatval($banked_row['total']);
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>View Till Report #<?= $report_id ?></title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body { background: #f8f9fa; font-size: 14px; }
    .card { margin-bottom: 1rem; }
    .summary { font-weight: bold; }
    .report-title { font-size: 18px; font-weight: bold; margin-bottom: 10px; }
    @media print {
      .no-print { display: none !important; }
      body { font-size: 11px; margin: 0; }
      .card { page-break-inside: avoid; }
    }
  </style>
</head>
<body>
<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="report-title">📄 Till Report #<?= $report_id ?></h5>
    <div class="no-print">
      <a href="list_till_reports.php" class="btn btn-secondary btn-sm">← Back to Reports</a>
      <a href="#" onclick="window.print()" class="btn btn-outline-primary btn-sm">🖨️ Print</a>
    </div>
  </div>

  <div class="card p-3">
    <div><strong>Branch:</strong> <?= htmlspecialchars($header['branch_name']) ?></div>
    <div><strong>Date:</strong> <?= $header['report_date'] ?></div>
    <div><strong>Remarks:</strong> <?= htmlspecialchars($header['remarks']) ?></div>
    <div><strong>Submitted On:</strong> <?= $header['created_at'] ?></div>
  </div>

  <?php
    $grandCash = $grandMpesa = $grandVisa = $grandCredit = $grandSTK = $grandOther = $grandTotal = 0;
    $totalDeclared = 0;

    foreach ($tills as $till):
        $tillDrop = $till['drop1'] + $till['drop2'] + $till['drop3'];
        $ttNotes = $till['note_1000'] + $till['note_500'] + $till['note_200'] + $till['note_100'] + $till['note_50'];
        $totalCash = $tillDrop + $ttNotes + $till['coins'];
        $totalAll = $totalCash + $till['mpesa'] + $till['visa'] + $till['credit'] + $till['stk'] + $till['other'];
        $autoOverShort = $totalAll - floatval($till['declared_total']);


        $grandCash += $totalCash;
        $grandMpesa += $till['mpesa'];
        $grandVisa += $till['visa'];
        $grandCredit += $till['credit'];
        $grandSTK += $till['stk'];
        $grandOther += $till['other'];
        $grandTotal += $totalAll;
        $totalDeclared += $till['declared_total'];
  ?>
  <div class="card">
    <div class="card-header bg-primary text-white">Till <?= htmlspecialchars($till['till_no']) ?></div>
    <div class="card-body">
      <div class="row">
        <div class="col-md-2"><strong>Batch No:</strong> <?= htmlspecialchars($till['batch_no']) ?></div>
        <div class="col-md-2"><strong>Cashier:</strong> <?= htmlspecialchars($till['cashier']) ?></div>
        <div class="col-md-2"><strong>Register:</strong> <?= htmlspecialchars($till['register']) ?></div>
        <div class="col-md-2"><strong>Opening:</strong> <?= number_format($till['opening'], 2) ?></div>
        <div class="col-md-2"><strong>Coins:</strong> <?= number_format($till['coins'], 2) ?></div>
        <div class="col-md-2"><strong>Manual Over/Short:</strong> <?= htmlspecialchars($till['over_short']) ?></div>
      </div>
      <hr class="my-2" />
      <div class="row">
        <div class="col-md-2"><strong>Drop Total:</strong> <?= number_format($tillDrop, 2) ?></div>
        <div class="col-md-2"><strong>TT Notes:</strong> <?= number_format($ttNotes, 2) ?></div>
        <div class="col-md-2"><strong>Total Cash:</strong> <?= number_format($totalCash, 2) ?></div>
        <div class="col-md-2"><strong>MPESA:</strong> <?= number_format($till['mpesa'], 2) ?></div>
        <div class="col-md-2"><strong>VISA:</strong> <?= number_format($till['visa'], 2) ?></div>
        <div class="col-md-2"><strong>CREDIT:</strong> <?= number_format($till['credit'], 2) ?></div>
      </div>
      <div class="row mt-2">
        <div class="col-md-2"><strong>STK Push:</strong> <?= number_format($till['stk'], 2) ?></div>
        <div class="col-md-2"><strong>Other:</strong> <?= number_format($till['other'], 2) ?></div>
        <div class="col-md-4 fw-bold text-success"><strong>Grand Total:</strong> <?= number_format($totalAll, 2) ?></div>
      </div>
      <hr class="my-2" />
      <div class="row">
        <div class="col-md-3"><strong>Declared Sales:</strong> <?= number_format($till['declared_total'], 2) ?></div>
        <div class="col-md-3"><strong>Auto Over/Short:</strong> 
          <span class="<?= $autoOverShort < 0 ? 'text-danger' : ($autoOverShort > 0 ? 'text-success' : '') ?>">
            <?= number_format($autoOverShort, 2) ?>
          </span>
        </div>
        <div class="col-md-6"><strong>Signature:</strong> <?= htmlspecialchars($till['signature']) ?></div>
      </div>

      <?php if (!empty($cheques_by_till[$till['till_no']])): ?>
        <hr>
        <h6 class="mt-2">🧾 Cheques Received</h6>
        <table class="table table-sm table-bordered">
          <thead class="table-light"><tr><th>Customer</th><th>Cheque No.</th><th>Amount</th></tr></thead>
          <tbody>
            <?php foreach ($cheques_by_till[$till['till_no']] as $chq): ?>
              <tr>
                <td><?= htmlspecialchars($chq['customer_name']) ?></td>
                <td><?= htmlspecialchars($chq['cheque_number']) ?></td>
                <td><?= number_format($chq['amount'], 2) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      <?php endif; ?>
    </div>
  </div>
  <?php endforeach; ?>

 <?php
$netOverShort = $grandTotal - $totalDeclared;
$netClass = $netOverShort < 0 ? 'text-danger' : ($netOverShort > 0 ? 'text-success' : 'text-muted');
$netPrefix = $netOverShort > 0 ? '+' : '';
?>
<div class="card border-warning">
  <div class="card-header bg-warning text-dark">
    📊 POS vs Cashier Sales Summary
    <span class="ms-2 small" data-bs-toggle="tooltip" title="Auto Over/Short = Cashier Counted - POS Expected">
      🛈
    </span>
  </div>
  <div class="card-body text-center summary">
    <div class="row">
      <div class="col-md-4">🖥️ Declared Sales (Expected by POS): <strong>KES <?= number_format($totalDeclared, 2) ?></strong></div>
      <div class="col-md-4">🧾 System Total (Cashier Counted): <strong>KES <?= number_format($grandTotal, 2) ?></strong></div>
      <div class="col-md-4">
        Auto Over/Short:
        <span class="<?= $netClass ?>">
          <?= $netPrefix ?>KES <?= number_format($netOverShort, 2) ?>
        </span>
      </div>
    </div>
  </div>
</div>


  <div class="card border-success">
    <div class="card-header bg-success text-white">🧮 Grand Summary</div>
    <div class="card-body">
      <div class="row text-center summary">
        <div class="col-md-2">Cash: KES <?= number_format($grandCash, 2) ?></div>
        <div class="col-md-2">MPESA: KES <?= number_format($grandMpesa, 2) ?></div>
        <div class="col-md-2">VISA: KES <?= number_format($grandVisa, 2) ?></div>
        <div class="col-md-2">CREDIT: KES <?= number_format($grandCredit, 2) ?></div>
        <div class="col-md-2">STK Push: KES <?= number_format($grandSTK, 2) ?></div>
        <div class="col-md-2">Other: KES <?= number_format($grandOther, 2) ?></div>
      </div>
      <div class="row text-center summary mt-2">
        <div class="col-md-4 text-success">+ Petty Receipts: KES <?= number_format($petty_receipts, 2) ?></div>
        <div class="col-md-4 text-primary">+ Cheques: KES <?= number_format($total_cheques, 2) ?></div>
        <div class="col-md-4 fw-bold text-dark">💰 Combined Total: KES <?= number_format($grandTotal + $petty_receipts + $total_cheques, 2) ?></div>
      </div>
    </div>
     <div class="card border-info">
        <div class="card-header bg-info text-white">💼 Petty Cash Transactions (<?= $report_date ?>)</div>
        <div class="card-body table-responsive">
            <?php if (count($petty_list) > 0): ?>
                <table class="table table-sm table-bordered small">
                    <thead class="table-light">
                        <tr>
                            <th>#</th>
                            <th>Type</th>
                            <th>Amount</th>
                            <th>Description</th>
                            <th>Reference No</th>
                            <th>Payee</th>
                            <th>Collected By</th>
                            <th>Status</th>
                            <th>Created By</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($petty_list as $i => $row): ?>
                            <tr>
                                <td><?= $i + 1 ?></td>
                                <td><?= htmlspecialchars($row['type']) ?></td>
                                <td><?= number_format($row['amount'], 2) ?></td>
                                <td><?= htmlspecialchars($row['description']) ?></td>
                                <td><?= htmlspecialchars($row['reference_no']) ?></td>
                                <td><?= htmlspecialchars($row['payee_name'] ?: '-') ?></td>
                                <td><?= htmlspecialchars($row['collected_by'] ?: '-') ?></td>
                                <td><?= htmlspecialchars($row['status']) ?></td>
                                <td><?= htmlspecialchars($row['full_name'] ?: '-') ?></td>
                            </tr>
                        <?php endforeach; ?>
                        <tr class="table-secondary fw-bold">
                            <td colspan="2">TOTALS</td>
                            <td><?= number_format($total_petty, 2) ?></td>
                            <td colspan="6">
                                Receipts: KES <?= number_format($total_receipt, 2) ?> |
                                Disbursements: KES <?= number_format($total_disbursement, 2) ?> |
                                Reimbursements: KES <?= number_format($total_reimbursement, 2) ?>
                            </td>
                        </tr>
                    </tbody>
                </table>
            <?php else: ?>
                <div class="alert alert-warning mb-0">No petty cash transactions recorded for this date.</div>
            <?php endif; ?>
        </div>
    </div>
</div>
  </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
  new bootstrap.Tooltip(document.body, {
    selector: '[data-bs-toggle="tooltip"]'
  });
</script>

</body>
</html>
