<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

ini_set('display_errors', 1);
error_reporting(E_ALL);

// Filters
$from = $_GET['from'] ?? date('Y-m-d', strtotime('-30 days'));
$to = $_GET['to'] ?? date('Y-m-d');
$branch_filter = $_GET['branch'] ?? '';
$bank_filter = $_GET['bank'] ?? '';

// WHERE clause
$where = "WHERE bd.deposit_date BETWEEN ? AND ?";
$params = [$from, $to];
$types = 'ss';

if ($branch_filter) {
    $where .= " AND bd.branch_name = ?";
    $params[] = $branch_filter;
    $types .= 's';
}
if ($bank_filter) {
    $where .= " AND bd.bank_name = ?";
    $params[] = $bank_filter;
    $types .= 's';
}

$sql = "SELECT bd.*, tr.report_date 
        FROM bank_deposits bd 
        JOIN till_report_headers tr ON bd.report_id = tr.id 
        $where ORDER BY bd.deposit_date DESC";

$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$deposits = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Metrics
$total = $high_value = 0;
foreach ($deposits as $d) {
    $total += $d['deposit_amount'];
    if ($d['deposit_amount'] >= 200000) $high_value++;
}
$count = count($deposits);
$avg = $count ? ($total / $count) : 0;
$high_percent = $count ? ($high_value / $count * 100) : 0;
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Bank Deposits | SmartPay360</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css" rel="stylesheet">
  <link href="https://cdn.datatables.net/buttons/2.4.1/css/buttons.bootstrap5.min.css" rel="stylesheet">
  <style>
    body { background-color: #f4f6fa; font-size: 13px; }
    .card-header { background: #007bff; color: white; }
    .summary-box { border-left: 5px solid #007bff; background: #eef7ff; padding: 12px 20px; border-radius: 5px; }
    .badge-high { background-color: #dc3545; }
    .cashier-icon { font-size: 12px; margin-right: 4px; color: #6c757d; }
    @media print {
      .no-print, .btn, .form-select, .form-control, .dataTables_filter, .dataTables_length, .dt-buttons { display: none !important; }
      .table { font-size: 11px; }
    }
  </style>
</head>
<body>

<div class="container py-4">
  <div class="card shadow-sm mb-4">
    <div class="card-header">
      <h5 class="mb-0"><i class="fas fa-university me-2"></i>Bank Deposits Overview</h5>
    </div>
    <div class="card-body">
      <div class="row text-center mb-3">
        <div class="col-md-4 summary-box">
          <strong>Total Deposits:</strong><br>
          <span class="text-primary fs-5">Ksh <?= number_format($total, 2) ?></span>
        </div>
        <div class="col-md-4 summary-box">
          <strong>Avg Deposit:</strong><br>
          <span class="text-success fs-5">Ksh <?= number_format($avg, 2) ?></span>
        </div>
        <div class="col-md-4 summary-box">
          <strong>High-Value (≥200K):</strong><br>
          <span class="text-danger fs-5"><?= $high_value ?> (<?= number_format($high_percent, 1) ?>%)</span>
        </div>
      </div>

      <form method="GET" class="row g-2 align-items-end no-print">
        <div class="col-md-3">
          <label>From:</label>
          <input type="date" name="from" value="<?= $from ?>" class="form-control">
        </div>
        <div class="col-md-3">
          <label>To:</label>
          <input type="date" name="to" value="<?= $to ?>" class="form-control">
        </div>
        <div class="col-md-3">
          <label>Branch:</label>
          <select name="branch" class="form-select">
            <option value="">All Branches</option>
            <?php
              $branches = $conn->query("SELECT branch_name FROM branches ORDER BY branch_name");
              while ($b = $branches->fetch_assoc()):
            ?>
            <option <?= $branch_filter === $b['branch_name'] ? 'selected' : '' ?>><?= htmlspecialchars($b['branch_name']) ?></option>
            <?php endwhile; ?>
          </select>
        </div>
        <div class="col-md-3">
          <label>Bank:</label>
          <select name="bank" class="form-select">
            <option value="">All Banks</option>
            <?php
              $banks = $conn->query("SELECT DISTINCT bank_name FROM bank_deposits ORDER BY bank_name");
              while ($b = $banks->fetch_assoc()):
            ?>
            <option <?= $bank_filter === $b['bank_name'] ? 'selected' : '' ?>><?= htmlspecialchars($b['bank_name']) ?></option>
            <?php endwhile; ?>
          </select>
        </div>
        <div class="col-md-2 mt-3">
          <button class="btn btn-primary w-100">🔍 Filter</button>
        </div>
      </form>
    </div>
  </div>

  <?php if (empty($deposits)): ?>
    <div class="alert alert-warning">⚠️ No bank deposits found.</div>
  <?php else: ?>
  <div class="card shadow-sm">
    <div class="card-body table-responsive">
      <table class="table table-bordered table-striped table-hover" id="depositsTable">
        <thead class="table-primary">
          <tr>
            <th>Report ID</th>
            <th>Branch</th>
            <th>Report Date</th>
            <th>Deposit Date</th>
            <th>Type</th>
            <th>Amount</th>
            <th>Bank</th>
            <th>Ref No</th>
            <th>By</th>
            <th>Remarks</th>
            <th>Slip</th>
            <th>Action</th>
          </tr>
        </thead>
        <tbody>
        <?php foreach ($deposits as $d): ?>
          <tr>
            <td><?= $d['report_id'] ?></td>
            <td><?= htmlspecialchars($d['branch_name']) ?></td>
            <td><?= $d['report_date'] ?></td>
            <td><?= $d['deposit_date'] ?></td>
            <td><?= htmlspecialchars($d['deposit_type']) ?></td>
            <td>
              Ksh <?= number_format($d['deposit_amount'], 2) ?>
              <?= $d['deposit_amount'] >= 200000 ? '<span class="badge badge-high ms-1">High</span>' : '' ?>
            </td>
            <td><?= htmlspecialchars($d['bank_name']) ?></td>
            <td><?= htmlspecialchars($d['reference_number']) ?></td>
            <td><i class="fas fa-user-circle cashier-icon"></i><?= htmlspecialchars($d['deposited_by']) ?></td>
            <td><?= htmlspecialchars($d['remarks']) ?></td>
            <td>
              <?php if (!empty($d['slip_path'])): ?>
                <a href="/smartpay360/<?= $d['slip_path'] ?>" target="_blank" class="btn btn-sm btn-outline-info">📄 View</a>
              <?php else: ?>
                <span class="text-muted">N/A</span>
              <?php endif; ?>
            </td>
            <td>
              <a href="view_report.php?report_id=<?= $d['report_id'] ?>" class="btn btn-sm btn-outline-secondary">🔍</a>
            </td>
          </tr>
        <?php endforeach; ?>
        </tbody>
        <tfoot class="table-light">
          <tr>
            <td colspan="5"><strong>Grand Total</strong></td>
            <td colspan="7"><strong>Ksh <?= number_format($total, 2) ?></strong></td>
          </tr>
        </tfoot>
      </table>
    </div>
  </div>
  <?php endif; ?>

  <div class="mt-3 no-print d-flex gap-2">
    <a href="record_deposit.php" class="btn btn-success btn-sm">➕ Record New Deposit</a>
    <a href="summaries.php" class="btn btn-secondary btn-sm">📊 Deposit Summaries</a>
    <a href="till_report.php" class="btn btn-secondary btn-sm">🧾 Create Till Report</a>
  </div>
</div>

<!-- Success Modal -->
<?php if (isset($_GET['success'])): ?>
<div class="modal fade" id="successModal" tabindex="-1" aria-labelledby="successModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content border-success">
      <div class="modal-header bg-success text-white">
        <h5 class="modal-title" id="successModalLabel">Success</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        ✅ <?= htmlspecialchars($_GET['success']) ?>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-success" data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>
<script>
  new bootstrap.Modal(document.getElementById('successModal')).show();
</script>
<?php endif; ?>

<script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.1/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.1/js/buttons.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.1/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.1/js/buttons.print.min.js"></script>
<script>
$(function() {
  $('#depositsTable').DataTable({
    dom: 'Bfrtip',
    buttons: ['copy', 'excel', 'pdf', 'print'],
    pageLength: 10,
    order: [[3, 'desc']]
  });
});
</script>
</body>
</html>
